/* xxerfc.h -- common erfc[fl] functionality */
#include "xmath.h"
_STD_BEGIN

FTYPE FFUN(erf)(FTYPE x);

#define ASIZE(x)	(sizeof (x) / sizeof ((x)[0]))
#define PSIZE(x)	(ASIZE(x) - 1)
#define POLY(x)		x, PSIZE(x)

	/* coefficients */
struct Approx {	/* description of one interval approximation */
	FTYPE to, mid;
	const FTYPE *num;
	int nsize;
	const FTYPE *den;
	int dsize;
	};

static const FTYPE mid00 = FLIT(24.0) / FLIT(32.0);
static const FTYPE mid0 = FLIT(32.0) / FLIT(32.0);
static const FTYPE mid01 = FLIT(41.0) / FLIT(32.0);

  #define DPOLY00(x)	FNAME(Poly)(x, POLY(c00))
  #define NPOLY00(x)	FNAME(Poly)(x, POLY(s00))

  #define DPOLY01(x)	FNAME(Poly)(x, POLY(c01))
  #define NPOLY01(x)	FNAME(Poly)(x, POLY(s01))

 #if   FBITS <= 24
static const FTYPE s00[] = {	/* 4/4, 0.47 <= x+3/4 < 1 */
	FLIT( 0.2065228066e-1),
	FLIT(-0.1509142800),
	FLIT( 0.4262061540),
	FLIT(-0.5573867797),
	FLIT( 0.2855036029),
	};

static const FTYPE c00[] = {
	FLIT( 0.52445813480e-1),
	FLIT( 0.69036908669e-1),
	FLIT( 0.42737154752),
	FLIT( 0.27041611799),
	FLIT( 0.98843403647),
	};

static const FTYPE s01[] = {	/* 4/4, 1 <= x+41/32 < 1.523 */
	FLIT( 0.1710573619e-1),
	FLIT(-0.8396320332e-1),
	FLIT( 0.1722734702),
	FLIT(-0.1720571890),
	FLIT( 0.6967651920e-1),
	};

static const FTYPE c01[] = {
	FLIT( 0.4467894790e-1),
	FLIT( 0.1527638176),
	FLIT( 0.5082267347),
	FLIT( 0.6499285695),
	FLIT( 0.9954897260),
	};

static const FTYPE s10[] = {	/* 7/1, 1.523 <= x < 1.75 */
	FLIT(-0.2239868484e-1),
	FLIT( 0.4179726125e-1),
	FLIT(-0.4815653097e-1),
	FLIT( 0.2894837499e-1),
	FLIT( 0.3064605465e-1),
	FLIT(-0.1470762643),
	FLIT( 0.4069878610),
	FLIT( 0.3017322315),
	};

static const FTYPE c10[] = {
	FLIT( 0.6926808746e-1),
	FLIT( 0.9999999985),
	};

static const FTYPE s11[] = {	/* 6/2, 1.75 <= x < 2.15 */
	FLIT(-0.8446285903e-2),
	FLIT( 0.3128812247e-1),
	FLIT(-0.7400038472e-1),
	FLIT( 0.1409578962),
	FLIT( 0.7004226993),
	FLIT( 0.8193149023),
	FLIT( 0.2627918549),
	};

static const FTYPE c11[] = {
	FLIT( 0.9353449747),
	FLIT( 1.5179787444),
	FLIT( 0.9986785175),
	};

static const FTYPE s12[] = {	/* 7/1, 2.15 <= x < 2.8 */
	FLIT( 0.1998715188),
	FLIT(-0.2797274548),
	FLIT( 0.1873345074),
	FLIT( 0.8898860043e-1),
	FLIT(-0.5291699014),
	FLIT( 1.0896799623),
	FLIT( 1.0596253814),
	FLIT( 0.2159547006),
	};

static const FTYPE c12[] = {
	FLIT( 2.7743459979),
	FLIT( 0.9999999888),
	};

static const FTYPE s13[] = {	/* 7/1, 2.8 <= x < 3.5 */
	FLIT( 0.1782164941),
	FLIT(-0.2101569589),
	FLIT( 0.1055503047),
	FLIT( 0.8748042990e-1),
	FLIT(-0.3116615231),
	FLIT( 0.5180952345),
	FLIT( 0.7383285367),
	FLIT( 0.1731499256),
	};

static const FTYPE c13[] = {
	FLIT( 1.4131173800),
	FLIT( 0.9999999940),
	};

static const FTYPE s14[] = {	/* 7/1, 3.5 <= x < 4.4 */
	FLIT( 0.2517430525),
	FLIT(-0.2327740672),
	FLIT( 0.6071824727e-1),
	FLIT( 0.1254376919),
	FLIT(-0.2350343518),
	FLIT( 0.2390976095),
	FLIT( 0.6255417894),
	FLIT( 0.1403571247),
	};

static const FTYPE c14[] = {
	FLIT( 0.7802650683),
	FLIT( 0.9999999786),
	};

static const FTYPE s15[] = {	/* 7/1, 4.4 <= x < 5.4 */
	FLIT( 0.4346081531),
	FLIT(-0.3009199102),
	FLIT(-0.2748935678e-1),
	FLIT( 0.2224760355),
	FLIT(-0.1563740339),
	FLIT(-0.1265105205),
	FLIT( 0.5353063469),
	FLIT( 0.1140211403),
	};

static const FTYPE c15[] = {
	FLIT( 0.3300570663e-1),
	FLIT( 0.9999999994),
	};

static const FTYPE s16[] = {	/* 5/3, 5.4 <= x < 8.0 */
	FLIT(-1.9548407228),
	FLIT( 5.8308227961),
	FLIT( 7.1402809645),
	FLIT( 7.4736829048),
	FLIT( 1.5645370870),
	FLIT( 0.8596663043e-1),
	};

static const FTYPE c16[] = {
	FLIT( 17.441400957),
	FLIT( 13.239545392),
	FLIT( 11.826775256),
	FLIT( 0.9940050704),
	};

static const FTYPE s17[] = {	/* 6/2, 8.0 <= x < 10.123 */
	FLIT( 0.2185443099),
	FLIT(-0.2082407401),
	FLIT(-0.2869901063),
	FLIT( 1.1248645477),
	FLIT( 0.6549960255),
	FLIT( 0.6194361398),
	FLIT( 0.6272644470e-1),
	};

static const FTYPE c17[] = {
	FLIT( 2.6221534281),
	FLIT( 1.0463543863),
	FLIT( 0.9997747361),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.75), FLIT(2515.0) 
		/ FLIT(4096.0), POLY(s10), POLY(c10)},
	{FLIT(2.15), FLIT(67931.0) 
		/ FLIT(131072.0), POLY(s11), POLY(c11)},
	{FLIT(2.8), FLIT(13795219.0) 
		/ FLIT(33554432.0), POLY(s12), POLY(c12)},
	{FLIT(3.5), FLIT(10785353.0) 
		/ FLIT(33554432.0), POLY(s13), POLY(c13)},
	{FLIT(4.4), FLIT(8606493.0) 
		/ FLIT(33554432.0), POLY(s14), POLY(c14)},
	{FLIT(5.4), FLIT(6919895.0) 
		/ FLIT(33554432.0), POLY(s15), POLY(c15)},
	{FLIT(8.0), FLIT(5204043.0) 
		/ FLIT(33554432.0), POLY(s16), POLY(c16)},
	{FLIT(10.123), FLIT(469311.0) 
		/ FLIT(4194304.0), POLY(s17), POLY(c17)},
	};

 #elif FBITS <= 53
#if defined(_BUILD_FOR_RENESAS) && defined(_R_FLT)	/* V10000K102 */
static const FTYPE s00[] = {	/* 8/9, 0.47 <= x+3/4 < 1  */
	FLIT( 2.0513042358524566e-04),
	FLIT(-1.7302909327917392e-03),
	FLIT( 5.1668629864273204e-03),
	FLIT(-6.1018108811685115e-03),
	FLIT( 1.7296128942169146e-02),
	FLIT(-1.2367859758895684e-01),
	FLIT( 3.7903680328736622e-01),
	FLIT(-5.2712243098316380e-01),
	FLIT( 2.8518837675284930e-01),
	};

static const FTYPE c00[] = {
	FLIT(-3.7788983802056160e-06),
	FLIT( 3.1563412406623647e-04),
	FLIT( 9.6597074108538763e-04),
	FLIT( 8.9003991783184362e-03),
	FLIT( 1.9304411411721720e-02),
	FLIT( 9.7000814701276833e-02),
	FLIT( 1.4001436609991327e-01),
	FLIT( 4.9370385730392385e-01),
	FLIT( 3.7276429728250793e-01),
	FLIT( 9.8734270070806918e-01),
	};

static const FTYPE s01[] = {	/* 8/9, 1 <= x+41/32 < 1.523 */
	FLIT( 7.7247320919989047e-06),
	FLIT( 2.2525341796674350e-05),
	FLIT(-1.9345571017397468e-04),
	FLIT(-2.3110094671285695e-03),
	FLIT( 2.2442956437592785e-02),
	FLIT(-8.2214574807852422e-02),
	FLIT( 1.5845614523514939e-01),
	FLIT(-1.6145785744410440e-01),
	FLIT( 6.9286736206390129e-02),
	};

static const FTYPE c01[] = {
	FLIT( 1.3243369303587245e-04),
	FLIT( 1.0961148685545738e-03),
	FLIT( 5.7965688990380855e-03),
	FLIT( 2.4145811406411855e-02),
	FLIT( 7.2705248947080026e-02),
	FLIT( 2.0197452162142901e-01),
	FLIT( 3.8821508541722938e-01),
	FLIT( 7.5162372253912657e-01),
	FLIT( 7.8397684142516699e-01),
	FLIT( 9.8992077724236471e-01),
	};

static const FTYPE s10[] = {	/* 13/1, 1.523 <= x < 2.15 */
	FLIT(-8.6381945584175987e-02),
	FLIT( 1.3483053545881871e-01),
	FLIT(-1.3245818902108339e-01),
	FLIT( 8.3163693902502264e-02),
	FLIT( 3.0111206414610810e-04),
	FLIT(-9.3853379462813039e-02),
	FLIT( 1.6426802871764809e-01),
	FLIT(-1.7233569546378735e-01),
	FLIT( 7.6415772074815064e-02),
	FLIT( 1.6395297850473245e-01),
	FLIT(-5.8471469916193586e-01),
	FLIT( 1.2144883338336665e+00),
	FLIT( 1.3717392508130257e+00),
	FLIT( 2.8071199775962147e-01),
	};

static const FTYPE c10[] = {
	FLIT( 3.4442009157955895e+00),
	FLIT( 9.9999999999999989e-01),
	};

static const FTYPE s11[] = {	/* 13/1, 2.15 <= x < 3.0 */
	FLIT(-2.4436063911004127e-01),
	FLIT( 2.8546331899100019e-01),
	FLIT(-1.9283511026461575e-01),
	FLIT( 5.7506411710317173e-02),
	FLIT( 5.5467128136468896e-02),
	FLIT(-1.1130874266260128e-01),
	FLIT( 1.0410616412706170e-01),
	FLIT(-4.8496478131438245e-02),
	FLIT(-3.0714623155242940e-02),
	FLIT( 1.0765310029967631e-01),
	FLIT(-1.6160396808256355e-01),
	FLIT( 1.8007923119004882e-01),
	FLIT( 6.3133425726592951e-01),
	FLIT( 2.1044604135810538e-01),
	};

static const FTYPE c11[] = {
	FLIT( 7.9057133939163815e-01),
	FLIT( 1.0000000000000000e+00),
	};

static const FTYPE s12[] = {	/* 13/1, 3.0 <= x < 4.0 */
	FLIT( 6.9402737093745861e+00),
	FLIT(-6.0854473462419341e+00),
	FLIT( 2.3187983173442523e+00),
	FLIT( 7.8118436567803518e-01),
	FLIT(-1.9139643568118463e+00),
	FLIT( 1.3773577786020115e+00),
	FLIT(-1.4960726890626716e-01),
	FLIT(-8.2875496207928301e-01),
	FLIT( 1.0098858543521478e+00),
	FLIT(-2.8824122606579655e-01),
	FLIT(-1.1216365186024262e+00),
	FLIT( 2.8575306094111910e+00),
	FLIT( 1.4558309276550108e+00),
	FLIT( 1.5830072471294548e-01),
	};

static const FTYPE c12[] = {
	FLIT( 6.0118008381220127e+00),
	FLIT( 9.9999999999999978e-01),
	};

static const FTYPE s13[] = {	/* 12/2, 4.0 <= x < 5.823 */
	FLIT( 1.9531849802619963e-01),
	FLIT( 6.1522573748464045e-02),
	FLIT(-3.6565091540205741e-01),
	FLIT( 3.4311325000620702e-01),
	FLIT(-3.1194495597758599e-02),
	FLIT(-2.8322613880361214e-01),
	FLIT( 3.3594849905314272e-01),
	FLIT(-3.5520022543526479e-02),
	FLIT(-5.3726815740653433e-01),
	FLIT( 1.2115303276048055e+00),
	FLIT( 1.3176530529685007e+00),
	FLIT( 7.7025064588641701e-01),
	FLIT( 1.1620354954329074e-01),
	};

static const FTYPE c13[] = {
	FLIT( 3.1334396699011919e+00),
	FLIT( 2.0717657133561196e+00),
	FLIT( 9.9760066369871014e-01),
	};

static const FTYPE s14[] = {	/* 13/1, 5.823 <= x < 10.123 */
	FLIT(-6.3311252399903765e+01),
	FLIT( 2.7778180026780866e+01),
	FLIT( 4.5223122806583538e+00),
	FLIT(-1.0063540545170136e+01),
	FLIT( 3.2656999777494211e+00),
	FLIT( 2.0232418241278265e+00),
	FLIT(-2.3303849802694638e+00),
	FLIT( 2.2776663101245270e-01),
	FLIT( 1.2444746449647448e+00),
	FLIT(-9.1481036444256325e-01),
	FLIT(-7.6409322595944351e-01),
	FLIT( 2.7683857465587320e+00),
	FLIT( 9.4493743012910891e-01),
	FLIT( 7.5631915367277106e-02),
	};

static const FTYPE c14[] = {
	FLIT( 5.2301252809321861e+00),
	FLIT( 9.9999999999999989e-01),
	};

static const FTYPE s15[] = {	/* 12/2, 10.123 <= x < 27.252 */
	 FLIT(-5.0452089243393763e+00),
	 FLIT( 4.9840891513090935e-01),
	 FLIT( 2.7489959219137470e+00),
	 FLIT(-9.3943527105192870e-01),
	 FLIT(-1.2368976662822648e+00),
	 FLIT( 8.9630338637280371e-01),
	 FLIT( 6.1024859928349151e-01),
	 FLIT(-9.9851426242736707e-01),
	 FLIT(-4.1379697483197009e-01),
	 FLIT( 2.7530374275544749e+00),
	 FLIT( 6.8214452414590943e-01),
	 FLIT( 5.9154026904535673e-01),
	 FLIT( 3.3307959299669690e-02)
	};

static const FTYPE c15[] = {
	 FLIT( 5.4661856147526686e+00),
	 FLIT( 9.7728961527023062e-01),
	 FLIT( 9.9575887705636490e-01)
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(2.150), FLIT( 5.0517556937292770e+15)
		/ FLIT( 9.0071992547409920e+15),	POLY(s10), POLY(c10)},
	{FLIT(3.000), FLIT( 7.1917947537854430e+15) 
		/ FLIT( 1.8014398509481984e+16), POLY(s11), POLY(c11)},
	{FLIT(4.000), FLIT( 2.6270997826327890e+15)
		/ FLIT( 9.0071992547409920e+15), POLY(s12), POLY(c12)},
	{FLIT(5.823), FLIT( 3.7986312158389470e+15) 
		/ FLIT( 1.8014398509481984e+16), POLY(s13), POLY(c13)},
	{FLIT(10.123), FLIT( 2.4366070867077830e+15)
		/ FLIT( 1.8014398509481984e+16), POLY(s14), POLY(c14)},
	{FLIT(27.252), FLIT( 1.0699196696489030e+15) 
		/ FLIT( 1.8014398509481984e+16), POLY(s15), POLY(c15)},
	};
#else
static const FTYPE s00[] = {	/* 8/9, 0.47 <= x+3/4 < 1  */
	FLIT( 0.20513042358524565923e-3),
	FLIT(-0.17302909327917391697e-2),
	FLIT( 0.51668629864273204744e-2),
	FLIT(-0.61018108811685117109e-2),
	FLIT( 0.17296128942169147008e-1),
	FLIT(-0.12367859758895683874),
	FLIT( 0.37903680328736621018),
	FLIT(-0.52712243098316381367),
	FLIT( 0.28518837675284931199),
	};

static const FTYPE c00[] = {
	FLIT(-0.37788983802056158995e-5),
	FLIT( 0.31563412406623646172e-3),
	FLIT( 0.96597074108538761276e-3),
	FLIT( 0.89003991783184363171e-2),
	FLIT( 0.19304411411721719127e-1),
	FLIT( 0.97000814701276839169e-1),
	FLIT( 0.14001436609991328054),
	FLIT( 0.49370385730392383516),
	FLIT( 0.37276429728250795375),
	FLIT( 0.98734270070806922458),
	};

static const FTYPE s01[] = {	/* 8/9, 1 <= x+41/32 < 1.523 */
	FLIT( 0.77247320919989038633e-5),
	FLIT( 0.22525341796674351121e-4),
	FLIT(-0.19345571017397467338e-3),
	FLIT(-0.23110094671285694424e-2),
	FLIT( 0.22442956437592785401e-1),
	FLIT(-0.82214574807852420520e-1),
	FLIT( 0.15845614523514938233),
	FLIT(-0.16145785744410440848),
	FLIT( 0.69286736206390129211e-1),
	};

static const FTYPE c01[] = {
	FLIT( 0.13243369303587245858e-3),
	FLIT( 0.10961148685545737546e-2),
	FLIT( 0.57965688990380852173e-2),
	FLIT( 0.24145811406411854685e-1),
	FLIT( 0.72705248947080019466e-1),
	FLIT( 0.20197452162142900622),
	FLIT( 0.38821508541722939820),
	FLIT( 0.75162372253912655342),
	FLIT( 0.78397684142516694323),
	FLIT( 0.98992077724236469320),
	};

static const FTYPE s10[] = {	/* 13/1, 1.523 <= x < 2.15 */
	FLIT(-0.86381945584175991307e-1),
	FLIT( 0.13483053545881869948),
	FLIT(-0.13245818902108337834),
	FLIT( 0.83163693902502265776e-1),
	FLIT( 0.30111206414610808633e-3),
	FLIT(-0.93853379462813035334e-1),
	FLIT( 0.16426802871764807892),
	FLIT(-0.17233569546378735938),
	FLIT( 0.76415772074815067543e-1),
	FLIT( 0.16395297850473244500),
	FLIT(-0.58471469916193587787),
	FLIT( 1.21448833383366656453),
	FLIT( 1.37173925081302581840),
	FLIT( 0.28071199775962147757),
	};

static const FTYPE c10[] = {
	FLIT( 3.44420091579558932594),
	FLIT( 0.99999999999999991413),
	};

static const FTYPE s11[] = {	/* 13/1, 2.15 <= x < 3.0 */
	FLIT(-0.24436063911004128091),
	FLIT( 0.28546331899100017862),
	FLIT(-0.19283511026461575004),
	FLIT( 0.57506411710317173766e-1),
	FLIT( 0.55467128136468893146e-1),
	FLIT(-0.11130874266260128482),
	FLIT( 0.10410616412706168628),
	FLIT(-0.48496478131438246512e-1),
	FLIT(-0.30714623155242940949e-1),
	FLIT( 0.10765310029967631882),
	FLIT(-0.16160396808256355249),
	FLIT( 0.18007923119004882982),
	FLIT( 0.63133425726592950391),
	FLIT( 0.21044604135810536956),
	};

static const FTYPE c11[] = {
	FLIT( 0.79057133939163814885),
	FLIT( 0.99999999999999999013),
	};

static const FTYPE s12[] = {	/* 13/1, 3.0 <= x < 4.0 */
	FLIT( 6.94027370937458631920),
	FLIT(-6.08544734624193419284),
	FLIT( 2.31879831734425252144),
	FLIT( 0.78118436567803517512),
	FLIT(-1.91396435681184626797),
	FLIT( 1.37735777860201142744),
	FLIT(-0.14960726890626714565),
	FLIT(-0.82875496207928303445),
	FLIT( 1.00988585435214778575),
	FLIT(-0.28824122606579654663),
	FLIT(-1.12163651860242613786),
	FLIT( 2.85753060941119083705),
	FLIT( 1.45583092765501070369),
	FLIT( 0.15830072471294549373),
	};

static const FTYPE c12[] = {
	FLIT( 6.01180083812201241500),
	FLIT( 0.99999999999999977751),
	};

static const FTYPE s13[] = {	/* 12/2, 4.0 <= x < 5.823 */
	FLIT( 0.19531849802619963394),
	FLIT( 0.61522573748464045868e-1),
	FLIT(-0.36565091540205739740),
	FLIT( 0.34311325000620700576),
	FLIT(-0.31194495597758597412e-1),
	FLIT(-0.28322613880361215939),
	FLIT( 0.33594849905314274848),
	FLIT(-0.35520022543526476069e-1),
	FLIT(-0.53726815740653433290),
	FLIT( 1.21153032760480563268),
	FLIT( 1.31765305296850066116),
	FLIT( 0.77025064588641700967),
	FLIT( 0.11620354954329073278),
	};

static const FTYPE c13[] = {
	FLIT( 3.13343966990119202182),
	FLIT( 2.07176571335611984074),
	FLIT( 0.99760066369871012713),
	};

static const FTYPE s14[] = {	/* 13/1, 5.823 <= x < 10.123 */
	FLIT(-63.3112523999037671752),
	FLIT( 27.7781800267808681065),
	FLIT( 4.52231228065835338299),
	FLIT(-10.0635405451701360167),
	FLIT( 3.26569997774942099066),
	FLIT( 2.02324182412782674568),
	FLIT(-2.33038498026946369464),
	FLIT( 0.22776663101245270494),
	FLIT( 1.24447464496474482245),
	FLIT(-0.91481036444256320588),
	FLIT(-0.76409322595944346933),
	FLIT( 2.76838574655873197941),
	FLIT( 0.94493743012910894854),
	FLIT( 0.75631915367277104834e-1),
	};

static const FTYPE c14[] = {
	FLIT( 5.23012528093218631775),
	FLIT( 0.99999999999999987376),
	};

static const FTYPE s15[] = {	/* 12/2, 10.123 <= x < 27.252 */
	 FLIT(-5.04520892433937595369),
	 FLIT( 0.49840891513090937302),
	 FLIT( 2.74899592191374698473),
	 FLIT(-0.93943527105192874028),
	 FLIT(-1.23689766628226479820),
	 FLIT( 0.89630338637280367301),
	 FLIT( 0.61024859928349147653),
	 FLIT(-0.99851426242736709386),
	 FLIT(-0.41379697483197006856),
	 FLIT( 2.75303742755447474047),
	 FLIT( 0.68214452414590938713),
	 FLIT( 0.59154026904535669166),
	 FLIT( 0.33307959299669689576e-1)
	};

static const FTYPE c15[] = {
	 FLIT( 5.46618561475266897367),
	 FLIT( 0.97728961527023062675),
	 FLIT( 0.99575887705636488663)
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(2.150), FLIT(5051755693729277.0)
		/ FLIT(9007199254740992.0),	POLY(s10), POLY(c10)},
	{FLIT(3.000), FLIT(7191794753785443.0) 
		/ FLIT(18014398509481984.0), POLY(s11), POLY(c11)},
	{FLIT(4.000), FLIT(2627099782632789.0)
		/ FLIT(9007199254740992.0), POLY(s12), POLY(c12)},
	{FLIT(5.823), FLIT(3798631215838947.0) 
		/ FLIT(18014398509481984.0), POLY(s13), POLY(c13)},
	{FLIT(10.123), FLIT(2436607086707783.0)
		/ FLIT(18014398509481984.0), POLY(s14), POLY(c14)},
	{FLIT(27.252), FLIT(1069919669648903.0) 
		/ FLIT(18014398509481984.0), POLY(s15), POLY(c15)},
	};
#endif												/* V10000K102 */

 #elif FBITS <= 64
static const FTYPE s00[] = {	/* 10/11, 0.47 <= x+3/4 < 1  */
	FLIT(-0.7018329417231617268285707e-5),
	FLIT( 0.5316624191864213621034350e-4),
	FLIT( 0.4489884606786366276312498e-5),
	FLIT(-0.1200083510953275192367988e-2),
	FLIT( 0.3930095206020320113427615e-2),
	FLIT(-0.1902949260656099221607933e-2),
	FLIT( 0.1258049641641308092136418e-3),
	FLIT(-0.7735953534496103538925258e-1),
	FLIT( 0.3131251047511031506969920),
	FLIT(-0.4888399778705276209621675),
	FLIT( 0.2849959128173297438132467),
	};

static const FTYPE c00[] = {
	FLIT( 0.4160581877887286615535647e-5),
	FLIT( 0.3204025433199894795245101e-4),
	FLIT( 0.2147122716188727503509631e-3),
	FLIT( 0.1100234052475893067122023e-2),
	FLIT( 0.4222805340965733436304633e-2),
	FLIT( 0.1641130511186100693773566e-1),
	FLIT( 0.4259871957284833524794042e-1),
	FLIT( 0.1315023613558032023364123),
	FLIT( 0.2253888405818472274988808),
	FLIT( 0.5583336132872461788281169),
	FLIT( 0.5038177541504158896701841),
	FLIT( 0.9866763767012900964025153),
	};

static const FTYPE s01[] = {	/* 9/9, 1 <= x+41/32 < 1.523 */
	FLIT(-0.2121931020770336378093494e-4),
	FLIT( 0.1687556131354152410395658e-3),
	FLIT(-0.4288026464282141266103981e-3),
	FLIT( 0.2334117001634438538309452e-3),
	FLIT(-0.2238360217710791110963296e-2),
	FLIT( 0.2388154060630853581953191e-1),
	FLIT(-0.8931329515751409636375287e-1),
	FLIT( 0.1692258863453691559852612),
	FLIT(-0.1673256772010857303866749),
	FLIT( 0.6940906385387910208650942e-1),
	};

static const FTYPE c01[] = {
	FLIT( 0.3092734275380921187678291e-4),
	FLIT( 0.4984587706260673309025593e-3),
	FLIT( 0.2745272882403183422380037e-2),
	FLIT( 0.1449071570757520258210030e-1),
	FLIT( 0.4590852965291016608004962e-1),
	FLIT( 0.1488398324208423083948529),
	FLIT( 0.2990164138754291971924870),
	FLIT( 0.6537859292053702531557075),
	FLIT( 0.7055983550620820871852924),
	FLIT( 0.9916685097307272464152380),
	};

static const FTYPE s10[] = {	/* 17/1, 1.523 <= x < 1.75 */
	FLIT(-0.9653955255771677358513551e-2),
	FLIT( 0.1603422611479822788613819e-1),
	FLIT(-0.1804913174395586449146470e-1),
	FLIT( 0.1562144548900286348699326e-1),
	FLIT(-0.9411874214714052372727867e-2),
	FLIT( 0.7997070910393863042894582e-3),
	FLIT( 0.8316119780928672052511990e-2),
	FLIT(-0.1577548803647596937029478e-1),
	FLIT( 0.1944458678092586832102814e-1),
	FLIT(-0.1750429099745858080149389e-1),
	FLIT( 0.8713752844746557245395708e-2),
	FLIT( 0.7371957830046197871529476e-2),
	FLIT(-0.3024572874960027395806095e-1),
	FLIT( 0.5836328101981261735852849e-1),
	FLIT(-0.8913249756615982517081792e-1),
	FLIT( 0.1189751041825559058379995),
	FLIT( 0.6149103785630582486759955),
	FLIT( 0.3017322407072364789374137),
	};

static const FTYPE c10[] = {
	FLIT( 0.7583642337480242785654188),
	FLIT( 0.9999999999999999999810835),
	};

static const FTYPE s11[] = {	/* 17/1, 1.75 <= x < 2.15 */
	FLIT(-0.2924011371507867960245615),
	FLIT( 0.4116836397165408878962838),
	FLIT(-0.3707435024758698623903798),
	FLIT( 0.2269154901384089185598952),
	FLIT(-0.4540585704720677887449808e-1),
	FLIT(-0.1126517621804647135964760),
	FLIT( 0.2030750745669572948758331),
	FLIT(-0.2059789292062607480164972),
	FLIT( 0.1275849392467581955343958),
	FLIT( 0.2534021993563346287371719e-2),
	FLIT(-0.1374126345357134908791138),
	FLIT( 0.2203806454175040091449009),
	FLIT(-0.1931213150792126210230306),
	FLIT( 0.3006379537685551255490765e-2),
	FLIT( 0.3910536355966172043281000),
	FLIT(-1.0146633967756646242597949),
	FLIT(-0.9937239387582718999909356e-1),
	FLIT( 0.2631395999350214183226129),
	};

static const FTYPE c11[] = {
	FLIT(-1.9753871130919731571032871),
	FLIT( 1.0000000000000000000113846),
	};

static const FTYPE s12[] = {	/* 16/2, 2.15 <= x < 2.8 */
	FLIT( 0.5750186138604141515934417e-3),
	FLIT( 0.1447999979195379196836404e-1),
	FLIT(-0.3825791773129098031523891e-1),
	FLIT( 0.5481624006806892078909377e-1),
	FLIT(-0.5173483871436975651860291e-1),
	FLIT( 0.2646187161318746101456982e-1),
	FLIT( 0.1298063542501993084024705e-1),
	FLIT(-0.5121379496090869497967055e-1),
	FLIT( 0.7042972191729125719810940e-1),
	FLIT(-0.5514348402752361396108203e-1),
	FLIT(-0.4419395327697799108001227e-2),
	FLIT( 0.1107595051799649592089361),
	FLIT(-0.2592559848942285470596586),
	FLIT( 0.4395879186015013766798202),
	FLIT( 1.2375877300466923494929335),
	FLIT( 0.9375539221464849219966151),
	FLIT( 0.2153644902608328647780499),
	};

static const FTYPE c12[] = {
	FLIT( 1.8754415944415655700351575),
	FLIT( 2.2149096076661499986934678),
	FLIT( 0.9972669518215459767725150),
	};

static const FTYPE s13[] = {	/* 17/1, 2.8 <= x < 3.5 */
	FLIT( 15.283533898258605821983881),
	FLIT(-13.965876090660972790934227),
	FLIT( 6.7601613284567686113729213),
	FLIT(-0.5798337745640578278269523),
	FLIT(-2.3260278956203666347553234),
	FLIT( 2.4508093407729257950252438),
	FLIT(-1.2475189263206310550125758),
	FLIT(-0.1159154238249767560070981e-1),
	FLIT( 0.6718138437239078090355353),
	FLIT(-0.6599057740180371402127718),
	FLIT( 0.2448682252941904450855265),
	FLIT( 0.2125912949456221186496405),
	FLIT(-0.4293508551074165647041102),
	FLIT( 0.2706202224318689473570068),
	FLIT( 0.2548267599109279208514024),
	FLIT(-1.0399357957701547353511803),
	FLIT( 0.1918393285511243553364634),
	FLIT( 0.1731499287561746916976512),
	};

static const FTYPE c13[] = {
	FLIT(-1.7430445460012103099601198),
	FLIT( 1.0000000000000000000134988),
	};

static const FTYPE s14[] = {	/* 17/1, 3.5 <= x < 4.4 */
	FLIT(-31.281696744383295294827562),
	FLIT( 24.216945488236361841135729),
	FLIT(-8.7754968023864070346983936),
	FLIT(-1.2001852071874533898020328),
	FLIT( 4.0670652269169314998497830),
	FLIT(-2.8605781451603900550348350),
	FLIT( 0.7493574482509169358140348),
	FLIT( 0.5872146297267657459699074),
	FLIT(-0.8520532682073156957088611),
	FLIT( 0.4572256924791850568298074),
	FLIT( 0.6393552359450871506877980e-1),
	FLIT(-0.3595097559771534846637548),
	FLIT( 0.3193062469542579315820448),
	FLIT(-0.7315799423941103361385503e-2),
	FLIT(-0.4371231667730439096731667),
	FLIT( 0.8767606601241418266635899),
	FLIT( 0.7989837633750439158169838),
	FLIT( 0.1403571419438127030090895),
	};

static const FTYPE c14[] = {
	FLIT( 2.0159838807988074937590016),
	FLIT( 0.9999999999999999999886454),
	};

static const FTYPE s15[] = {	/* 17/1, 4.4 <= x < 5.0 */
	FLIT(-151.21104348416409423008325),
	FLIT( 92.087224786639440802160942),
	FLIT(-18.000908453158076914219693),
	FLIT(-13.172610703302200528585692),
	FLIT( 13.165183781488721961110880),
	FLIT(-4.3872383778374523189578412),
	FLIT(-1.3190097925780893593510249),
	FLIT( 2.2970790740645687529866700),
	FLIT(-1.0049186689522879526503461),
	FLIT(-0.2459315906836111175560431),
	FLIT( 0.5996644826605351243781802),
	FLIT(-0.2817663698641873201900732),
	FLIT(-0.1519782717656607350257868),
	FLIT( 0.3054497089430299092428441),
	FLIT(-0.7754024657230570920920737e-1),
	FLIT(-0.4173972853784075966013266),
	FLIT( 0.4729083011519756165437233),
	FLIT( 0.1140211480926708498173065),
	};

static const FTYPE c15[] = {
	FLIT(-0.5142440193425373705385087),
	FLIT( 1.0000000000000000000037076),
	};

static const FTYPE s16[] = {	/* 17/1, 5.0 <= x < 5.5 */
	FLIT( 1274.2311034659763798200708),
	FLIT(-682.07929095410118091097216),
	FLIT( 66.524785666473135311415960),
	FLIT( 133.12251618430016290174113),
	FLIT(-91.117085644405799886223576),
	FLIT( 13.543875444367988405508585),
	FLIT( 19.795434889441528636188198),
	FLIT(-15.769207398829962489918841),
	FLIT( 2.1098587579267861925769546),
	FLIT( 5.0202425936724555957295921),
	FLIT(-4.0197720071058168299779048),
	FLIT(-0.9679391895066410324017424e-1),
	FLIT( 2.6267567605737897544878500),
	FLIT(-1.8039749579423652915401630),
	FLIT(-1.6540259699437828527292000),
	FLIT( 6.0164454032121013344887705),
	FLIT( 1.6595312711433743824389695),
	FLIT( 0.9832407872125064369361241e-1),
	};

static const FTYPE c16[] = {
	FLIT( 11.390481788267691975881558),
	FLIT( 0.9999999999999999997957569),
	};

static const FTYPE s17[] = {	/* 17/1, 5.5 <= x < 8.0 */
	FLIT( 478.76968739745763802905195),
	FLIT(-234.26766162665630072824468),
	FLIT( 10.860479601482098620918599),
	FLIT( 47.980993822524792276650838),
	FLIT(-27.031688063535239801996059),
	FLIT( 1.5054938927412813761837822),
	FLIT( 6.7585247785637749497998827),
	FLIT(-4.0661405224132287427363511),
	FLIT(-0.3708410506635854721895086e-1),
	FLIT( 1.5387069346097967895894101),
	FLIT(-0.8805585996291039085158500),
	FLIT(-0.2283867248583530846721426),
	FLIT( 0.6836438584795399508773077),
	FLIT(-0.3141850583901823835491683),
	FLIT(-0.5156900199927111481399804),
	FLIT( 1.3591082957040177218208959),
	FLIT( 0.7792544755417622512618992),
	FLIT( 0.8648511635875774354840146e-1),
	};

static const FTYPE c17[] = {
	FLIT( 2.7090238482726072576576194),
	FLIT( 0.9999999999999999999537675),
	};

static const FTYPE s18[] = {	/* 17/1, 8.0 <= x < 10.123 */
	FLIT( 1861.0789300250465036332892),
	FLIT(-631.55435577053254534611948),
	FLIT(-113.84163484817081526679330),
	FLIT( 156.67739648824604689419958),
	FLIT(-31.061583825042772353084159),
	FLIT(-22.761252101613483643216849),
	FLIT( 15.030082850585562645233039),
	FLIT( 0.5260768824948086112297659),
	FLIT(-4.3657447584184482809481060),
	FLIT( 1.3619720772950229287509571),
	FLIT( 1.0774738999524464137692507),
	FLIT(-0.9491628798244131641238829),
	FLIT(-0.2142409764913429830321848),
	FLIT( 0.7043399471447166291411354),
	FLIT(-0.3953470086917741754088738e-1),
	FLIT(-1.3000003211200829592314173),
	FLIT( 0.4167587582845941416015837),
	FLIT( 0.6274058386619807100001339e-1),
	};

static const FTYPE c18[] = {
	FLIT(-2.1860376258435981182263118),
	FLIT( 1.0000000000000000000493315),
	};

static const FTYPE s19[] = {	/* 17/1, 10.123 <= x < 15 */
	FLIT(-3242.5621990337150867145466),
	FLIT( 884.83949984699839157842083),
	FLIT( 295.44276787164746777568462),
	FLIT(-238.63083642642617119757398),
	FLIT( 11.419827285980128995807490),
	FLIT( 45.672538160358752409976399),
	FLIT(-15.847330136910384789530694),
	FLIT(-6.5559451546326645642500183),
	FLIT( 6.3460436471159534609128175),
	FLIT( 0.5914851883770026233593104e-1),
	FLIT(-2.3042387836235806401170763),
	FLIT( 0.7590980884141580072641557),
	FLIT( 0.9451810946992212896315443),
	FLIT(-0.9149229564111930568196245),
	FLIT(-0.5392611201203729368799627),
	FLIT( 2.2792869134505805191858059),
	FLIT( 0.7539662756545708841630145),
	FLIT( 0.4651494517458136536869452e-1),
	};

static const FTYPE c19[] = {
	FLIT( 4.2023376250003206168882070),
	FLIT( 0.9999999999999999998975740),
	};

static const FTYPE s20[] = {	/* 19/1, 15 <= x < 50 */
	FLIT(-247173.05040492535461637350),
	FLIT( 33530.552881159743183528307),
	FLIT( 23788.793175484248041756528),
	FLIT(-7201.7393455073430967679832),
	FLIT(-1953.0854968170879020365039),
	FLIT( 1216.2147723421575289945249),
	FLIT( 103.02505233418718784023825),
	FLIT(-195.95792794841849284389420),
	FLIT( 11.213284912639695255037466),
	FLIT( 32.338084692627528235681983),
	FLIT(-6.7792123167142865945278414),
	FLIT(-5.6951232446764475171136183),
	FLIT( 2.3997359069765220778109328),
	FLIT( 1.1001242122049129044198355),
	FLIT(-0.8752191842583602035892585),
	FLIT(-0.2331173101730793468166244),
	FLIT( 0.3977554735356351359048166),
	FLIT( 0.4320103288731904402126176e-1),
	FLIT(-0.2803095410295435265475986),
	FLIT( 0.5707526039500161702440413e-1),
	FLIT( 0.5666630608262658646920605),
	FLIT( 0.2442532548205654959722473e-1),
	};

static const FTYPE c20[] = {
	FLIT( 0.1660243910883325199609841),
	FLIT( 0.9999999999999999999982600),
	};

static const FTYPE s21[] = {	/* 16/2, 50 <= x < 108 */
	FLIT(-116.49768714835804279654658),
	FLIT( 83.391726226342121709282798),
	FLIT( 32.637162297297772852643023),
	FLIT(-31.581112239042922270509673),
	FLIT(-7.9476515299099230286885946),
	FLIT( 10.833571170206052251104598),
	FLIT( 2.0564936461794721810406782),
	FLIT(-4.2081093326695049355429626),
	FLIT(-0.6238331211894221419597367),
	FLIT( 2.1295530386868463107823787),
	FLIT( 0.2449982599565170930052298),
	FLIT(-1.7143423101304096449402970),
	FLIT(-0.1475571522256863022101620),
	FLIT( 4.0080437944585646652924984),
	FLIT( 0.2381965308694576032756864),
	FLIT( 0.5661165923576962622152371),
	FLIT( 0.6203797325885194556242073e-2),
	};

static const FTYPE c21[] = {
	 FLIT( 7.6102476522025989061778159),
	 FLIT( 0.3550340584992982258925006),
	 FLIT( 0.9996917849700857947389877),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.75), FLIT(22653106970547364201.0)
		/ FLIT(36893488147419103232.0), POLY(s10), POLY(c10)},
	{FLIT(2.15), FLIT(298763712157256027.0)
		/ FLIT(576460752303423488.0), POLY(s11), POLY(c11)},
	{FLIT(2.8), FLIT(3792000961995941881.0)
		/ FLIT(9223372036854775808.0), POLY(s12), POLY(c12)},
	{FLIT(3.5), FLIT(5929310595120927305.0)
		/ FLIT(18446744073709551616.0), POLY(s13), POLY(c13)},
	{FLIT(4.4), FLIT(591433758856759163.0)
		/ FLIT(2305843009213693952.0), POLY(s14), POLY(c14)},
	{FLIT(5.0), FLIT(7608505552287609673.0)
		/ FLIT(36893488147419103232.0), POLY(s15), POLY(c15)},
	{FLIT(5.5), FLIT(6527309441466456725.0)
		/ FLIT(36893488147419103232.0), POLY(s16), POLY(c16)},
	{FLIT(8.0), FLIT(5721906726567314621.0)
		/ FLIT(36893488147419103232.0), POLY(s17), POLY(c17)},
	{FLIT(10.123), FLIT(1032025902795114479.0)
		/ FLIT(9223372036854775808.0), POLY(s18), POLY(c18)},
	{FLIT(15.0), FLIT(3052043540214067405.0)
		/ FLIT(36893488147419103232.0), POLY(s19), POLY(c19)},
	{FLIT(50.0), FLIT(1598717819721494473.0)
		/ FLIT(36893488147419103232.0), POLY(s20), POLY(c20)},
	{FLIT(108.0), FLIT(405828369621610135.0)
		/ FLIT(36893488147419103232.0), POLY(s21), POLY(c21)},
	};

#elif FBITS <= 113
static const FTYPE s00[] = {	/* 13/12, 0.47 <= x+3/4 < 1  */
	FLIT( 0.52754218101397244528203958370699099e-6),
	FLIT(-0.53528772961612346236819345027293013e-5),
	FLIT( 0.16559076171869713756505852246688323e-4),
	FLIT(-0.14498462539219643684569109278725537e-4),
	FLIT( 0.20319697543527248631514032009337209e-3),
	FLIT(-0.18643240922827289625177663404261420e-2),
	FLIT( 0.58976379414957810818252236705957240e-2),
	FLIT(-0.70428259583422826208507864398069104e-2),
	FLIT( 0.12363055112016324040297324781490474e-1),
	FLIT(-0.99701086219911386468881562076227439e-1),
	FLIT( 0.33774844810529933956486309267306220),
	FLIT(-0.50059082056276792691965950727482489),
	FLIT( 0.28491289679252547779564282567238028),
	};

static const FTYPE c00[] = {
	FLIT( 0.86297975568425441872615309943641538e-6),
	FLIT( 0.49113355631578292152721189901488996e-5),
	FLIT( 0.51782784952936356969301701303074660e-4),
	FLIT( 0.20873941954781257893888633719702743e-3),
	FLIT( 0.12702219327812972160923315920305853e-2),
	FLIT( 0.38405482213549864685292461728523568e-2),
	FLIT( 0.16934232270147787887323877427488759e-1),
	FLIT( 0.38113707319331918052579911573029877e-1),
	FLIT( 0.13067594145804442729348872057377841),
	FLIT( 0.20308567194080855726059368617529748),
	FLIT( 0.55208365719882539943005691366952727),
	FLIT( 0.46249575771737794465145494341392756),
	FLIT( 0.98638896924427677981705978617476407),
	};

static const FTYPE s01[] = {	/* 19/13, 1 <= x+41/32 < 1.523 */
	FLIT( 0.11809613196628041957749867207508018e-6),
	FLIT(-0.14653445245488089708988119584608428e-5),
	FLIT( 0.49122684137474296932203404218206340e-5),
	FLIT( 0.17552166936882830642524501788821637e-4),
	FLIT(-0.18310143160834672924955425090052953e-3),
	FLIT( 0.47508618655369592713493576148335939e-3),
	FLIT( 0.16157930571154245132464324728468401e-4),
	FLIT(-0.91110947481098322745932176976355770e-3),
	FLIT(-0.11676029247089478409346388372113838e-1),
	FLIT( 0.72466718082419136223547336683417777e-1),
	FLIT(-0.18967104397626032176342619584286389),
	FLIT( 0.27264511600606091600853812667388099),
	FLIT(-0.21169068110972473844240294228310973),
	FLIT( 0.69675613869555210035221146210047740e-1),
	};

static const FTYPE c01[] = {
	FLIT(-0.13093132896534888149121455875737355e-6),
	FLIT(-0.13509420436661031092598705490268308e-5),
	FLIT(-0.10945852727479483762818246018530927e-4),
	FLIT(-0.64397518289070908735147547993048392e-4),
	FLIT(-0.30450054692965143485544563992983231e-3),
	FLIT(-0.12671533893639817714161461592589284e-2),
	FLIT(-0.41264466870689710865541901560675944e-2),
	FLIT(-0.12941947290437691106673265026941327e-1),
	FLIT(-0.28399134929533632334750398931392818e-1),
	FLIT(-0.67847051053516971162374917631533731e-1),
	FLIT(-0.72424700341629639187695812634583387e-1),
	FLIT(-0.13505137042973807764946160428430364),
	FLIT( 0.17420370482430153102419278518053199),
	FLIT( 0.83632370038166539711527418184073591e-1),
	FLIT( 0.99547679127405273304083278913668136)
	};

static const FTYPE s10[] = {	/* 30/1, 1.523 <= x < 1.75 */
	FLIT( 0.34755393245650761550157419722584964e-1),
	FLIT(-0.54499181669025718254919445649020140e-1),
	FLIT( 0.59344712632243326407770236246179947e-1),
	FLIT(-0.52588372367841942012526368858930599e-1),
	FLIT( 0.38032193048676581928010862705263136e-1),
	FLIT(-0.19848747188404581835971166873749048e-1),
	FLIT( 0.19056793547013475236683462909450704e-2),
	FLIT( 0.12739950007748054781732502302945770e-1),
	FLIT(-0.22134754330191452481338968840798483e-1),
	FLIT( 0.25519734432897095547794664207042790e-1),
	FLIT(-0.23245964928112710839048629631385024e-1),
	FLIT( 0.16556416750232010873326555092640552e-1),
	FLIT(-0.72817860592549918094702925873864308e-2),
	FLIT(-0.25013569038914991694887778654088089e-2),
	FLIT( 0.10806257388691196252422404399689507e-1),
	FLIT(-0.16029569192211167153340552095378289e-1),
	FLIT( 0.17176799857647430921639863069190130e-1),
	FLIT(-0.14004107008763216344758491091195141e-1),
	FLIT( 0.70710800598252595642912002764977250e-2),
	FLIT( 0.22919510424671619858958162044751779e-2),
	FLIT(-0.12091394690274192943328313765909034e-1),
	FLIT( 0.19836875273311668593522621250801917e-1),
	FLIT(-0.22747909501822802078598311876974530e-1),
	FLIT( 0.17980934495086888553605061943490629e-1),
	FLIT(-0.28575001812153028599013500591804491e-2),
	FLIT(-0.24923536687608549227824833646523599e-1),
	FLIT( 0.67103969463937690590370385667514900e-1),
	FLIT(-0.12472488869609027040090094894203917),
	FLIT( 0.19803269765856038583794798269383939),
	FLIT( 0.67669489267111547575994375864866836),
	FLIT( 0.30173224070723647895275211466921615),
	};

static const FTYPE c10[] = {
	FLIT( 0.96313026724593361214485391412572241),
	FLIT( 0.99999999999999999999999999999999994),
	};

static const FTYPE s11[] = {	/* 30/1, 1.75 <= x < 2.15 */
	FLIT(-4.06772841150357778999148299519040717),
	FLIT( 5.43008493219639559060328895265639206),
	FLIT(-4.85563930696023227620008496136256838),
	FLIT( 3.36646141179064879585356991379818341),
	FLIT(-1.67145246684972576501073369349658121),
	FLIT( 0.24084354961686431000615784896262755),
	FLIT( 0.69554117575115622031757719309949643),
	FLIT(-1.10474329435432730658283736313778394),
	FLIT( 1.08360748607821851332740690288897249),
	FLIT(-0.79089887692677209699732989027116442),
	FLIT( 0.39173687085209756252532533531475308),
	FLIT(-0.20295369737729829811219407553787818e-1),
	FLIT(-0.23870946046913517602618105941923705),
	FLIT( 0.35271732185219581177788297297081782),
	FLIT(-0.33289983146789144444629653416206209),
	FLIT( 0.21963447480233512685083190954790083),
	FLIT(-0.66290905662888766959753935714627772e-1),
	FLIT(-0.75364929898832557820300502306341834e-1),
	FLIT( 0.16610751566664091011697259406283792),
	FLIT(-0.18566208480424629633988520222931208),
	FLIT( 0.13466073302405969456297634630499509),
	FLIT(-0.32861393573215428666868332404924842e-1),
	FLIT(-0.85300352160430508604076755946552632e-1),
	FLIT( 0.17665322767104649162109823347122426),
	FLIT(-0.19563610485342651844332505647699902),
	FLIT( 0.10016115229066906911508507333805643),
	FLIT( 0.14363063383631101714759382365649138),
	FLIT(-0.55807596284651055307526765635159805),
	FLIT( 1.15226836726804900820035974805190081),
	FLIT( 1.25687076740030871294059656719759617),
	FLIT( 0.26313959993502141832204021632020846),
	};

static const FTYPE c11[] = {
	FLIT( 3.17869521283328044342452602846792824),
	FLIT( 0.99999999999999999999999999999999998),
	};

static const FTYPE s12[] = {	/* 30/1, 2.15 <= x < 2.8 */
	FLIT(-239.656120048281304284588009240600681),
	FLIT( 258.533686130895772943390732625741660),
	FLIT(-177.218517228388431435970270122192570),
	FLIT( 86.1471894141729832215939787908698517),
	FLIT(-19.9169623877442604880701014247051405),
	FLIT(-14.7838155747411627611931854145850632),
	FLIT( 24.8567742013962782618796733836244713),
	FLIT(-20.9660343786206704194703467965000723),
	FLIT( 12.1621342792376311321078401126763343),
	FLIT(-4.02563189120877911372936201220888404),
	FLIT(-1.12262148383250690956597744404200472),
	FLIT( 3.16025749463071296772625269054704198),
	FLIT(-3.02584816817609048110312011634694675),
	FLIT( 1.87340393738200171914559209193281048),
	FLIT(-0.60532769213076375204322716899600143),
	FLIT(-0.28110742603217404002728562581392579),
	FLIT( 0.65099726225994907702440909233509937),
	FLIT(-0.60319288443385460002686144454485789),
	FLIT( 0.33371250322485492270008669409846368),
	FLIT(-0.32725204727288379930747929506163218e-1),
	FLIT(-0.17222718529242278878043726703915712),
	FLIT( 0.23204871189946006817043319898535714),
	FLIT(-0.16424781207296375475640854383173240),
	FLIT( 0.27738234010939345933643044930868256e-1),
	FLIT( 0.10430061731106307638155386353071254),
	FLIT(-0.16635739104611475901270480825672175),
	FLIT( 0.11421233841562259217907001491291263),
	FLIT( 0.69836443885421988721231571592476916e-1),
	FLIT(-0.37844001660960956071060818903814809),
	FLIT( 0.37112874852101203964501481113331127),
	FLIT( 0.21595470487361627792857054388581593),
	};

static const FTYPE c12[] = {
	FLIT(-0.41380700979244557247471330757347947),
	FLIT( 1.0),
//	FLIT( 1.00000000000000000000000000000000002),
	};

static const FTYPE s13[] = {	/* 31/2, 2.8 <= x < 3.5 */
	FLIT( 346.439880210733860486941606614549176),
	FLIT(-540.319356902337284257411288377018973),
	FLIT( 476.326833011261504032473437277349150),
	FLIT(-277.362366930687340270791379432793974),
	FLIT( 83.3378698515477606775063145653891390),
	FLIT( 34.7458359508188677195067713192348076),
	FLIT(-73.0731818660835925229711602760784421),
	FLIT( 61.2111308688331277819703136667890667),
	FLIT(-31.9667118589832412486681294835619843),
	FLIT( 6.59109887468156855189481264526636434),
	FLIT( 7.08231030646981883252264243042202607),
	FLIT(-10.1686646901642951422898981754469847),
	FLIT( 7.30450392728424854181784446250101338),
	FLIT(-2.90927443639598048068970221385289032),
	FLIT(-0.39335320131477889932677269206366209),
	FLIT( 1.81018312351055173066838445084408348),
	FLIT(-1.71053867222146406323234633524928898),
	FLIT( 0.88299950516158239705534068897656521),
	FLIT(-0.24924819418952448800842476094137586e-1),
	FLIT(-0.47668487046480586371039915006503938),
	FLIT( 0.54645260820150047584673521634291135),
	FLIT(-0.31272224363234211905312911128375742),
	FLIT(-0.20519627409674478160156233536839282e-1),
	FLIT( 0.27108117231064216437508228278638914),
	FLIT(-0.32903724495242017779421499616065902),
	FLIT( 0.16592292091163408052682789437926510),
	FLIT( 0.18228969480874988062487965393882211),
	FLIT(-0.64289945219843184275683734422668452),
	FLIT( 1.13340067202101042065984075129130028),
	FLIT( 1.94917744099213106050019692372811860),
	FLIT( 1.02461406692692302699724329115832216),
	FLIT( 0.17276005998091084307864466177574769),
	};

static const FTYPE c13[] = {
	FLIT( 3.53054860608433595881716299251913157),
	FLIT( 3.07293345243468303545481648979152969),
	FLIT( 0.99774837461346662247524415625477087),
	};

static const FTYPE s14[] = {	/* 32/1, 3.5 <= x < 4.4 */
	FLIT(-5699331.74550098147670428718517101243),
	FLIT( 4031837.03669054699220517477008403045),
	FLIT(-1571148.50016757809485023539861925557),
	FLIT( 223459.013376888443658104924720296447),
	FLIT( 194148.873162445098414432718805582788),
	FLIT(-189777.458402518686658602000798667365),
	FLIT( 90676.0013916092496078062409054404459),
	FLIT(-20059.8696337309414237407252136852618),
	FLIT(-7528.70863083528006087455834741290515),
	FLIT( 10419.9008628264940842205772799661935),
	FLIT(-5659.72222368517043706858482199201277),
	FLIT( 1419.71469574807487186513379054351892),
	FLIT( 470.190856841608945133735311812585256),
	FLIT(-735.723638198213173263273358406296282),
	FLIT( 411.220691671325042379085198889481878),
	FLIT(-90.7807440390278138864880792623416628),
	FLIT(-56.6516285775055160871696476016954771),
	FLIT( 71.2614090213842693673351896138260151),
	FLIT(-35.4403510339140903509037590896200657),
	FLIT( 2.54691556164177608816693195665733279),
	FLIT( 10.9086447256677484002208391170719220),
	FLIT(-9.49012579251281264407217194545297378),
	FLIT( 2.99463100933309183260425856429239736),
	FLIT( 1.83048881372788008627560632171796920),
	FLIT(-2.95371501310687295751674367972977466),
	FLIT( 1.46498748252721515092002659712803591),
	FLIT( 0.58123053312330494535870880995285425),
	FLIT(-1.60012706871487540481252876098403381),
	FLIT( 0.97807739371717327360864633795205345),
	FLIT( 1.06292736424941136625758767221260947),
	FLIT(-3.85643924918790748013027895285857024),
	FLIT(-0.48842883968800199505637596629463741),
	FLIT( 0.14035714194381270301358945393523909),
	};

static const FTYPE c14[] = {
	FLIT(-7.15642149333184113201069940618059388),
	FLIT( 1.0),
//	FLIT( 1.00000000000000000000000000000000033),
	};

static const FTYPE s15[] = {	/* 30/1, 4.4 <= x < 5.4 */
	FLIT( 4234088.44764765041793834585327203739),
	FLIT(-2506311.03777968292094811325270858200),
	FLIT( 712149.052169121759809787998154153425),
	FLIT( 36641.3060250398295324725380957780761),
	FLIT(-156228.766392342582232681041591113020),
	FLIT( 87080.6058797916351099021270557119506),
	FLIT(-21844.7994701902150755601495972634336),
	FLIT(-4358.62347891408508009318584106642188),
	FLIT( 7207.63978027544325329702953436869867),
	FLIT(-3492.52984399940283407002395802576326),
	FLIT( 565.548422044596079667019101544959408),
	FLIT( 445.211065361584117839774469802269273),
	FLIT(-411.978623998351018250890973156086834),
	FLIT( 150.224424517998987932229139725032976),
	FLIT( 9.96239880644816173415095425950500115),
	FLIT(-46.5429521167660897626998163631965283),
	FLIT( 27.3003259229680027122943151115782927),
	FLIT(-4.05515754083624819852769206164211991),
	FLIT(-5.90767405212392770585678175589377970),
	FLIT( 5.33491058444676008219979746603995298),
	FLIT(-1.50065984537113919300294491935892283),
	FLIT(-1.06797619381486802062776794292068003),
	FLIT( 1.42335218264609754958785330407025566),
	FLIT(-0.52351524418705791656850593784129177),
	FLIT(-0.41289926187946185536947627192168144),
	FLIT( 0.70036378096567131787751555629388198),
	FLIT(-0.26265033977228354492357081006227157),
	FLIT(-0.61729542420850042278597680398844171),
	FLIT( 1.57423174803127747808277063663139043),
	FLIT( 0.90013212171810841142671907044758209),
	FLIT( 0.11402114809267084982071593485386670),
	};

static const FTYPE c15[] = {
	FLIT( 3.23263827146641000842980653488579967),
	FLIT( 0.99999999999999999999999999999999987),
	};

static const FTYPE s16[] = {	/* 29/2, 5.4 <= x < 6.15 */
	FLIT(-2408962.42446692429298461421543199110),
	FLIT( 759214.926821913910085743009919673106),
	FLIT( 449137.796047460600710651285905856187),
	FLIT(-430259.878371536534632481754923384860),
	FLIT( 111869.814684863164587447406905697475),
	FLIT( 38209.7719285932451492308213520003638),
	FLIT(-41932.2459693300317946804048608015986),
	FLIT( 12129.6570376114437654408936952027202),
	FLIT( 2961.15961878475927128899089642976572),
	FLIT(-3958.29340701458485788545367071640813),
	FLIT( 1233.82387378332377729157815437940403),
	FLIT( 292.610393013528048355307032319362239),
	FLIT(-421.536816901133484726959509833653958),
	FLIT( 130.945701048145066391075663207618111),
	FLIT( 42.7745276194331740963381571438891294),
	FLIT(-55.3056133014253739825139290518820057),
	FLIT( 14.9876105238649370611917711853008852),
	FLIT( 9.28868735787810781605615603818585652),
	FLIT(-9.59238841883845961894980039769063260),
	FLIT( 1.58962614610848707344930642652180714),
	FLIT( 3.01669012241779536776375610826045658),
	FLIT(-2.38079746777827843055765966178057361),
	FLIT(-0.25167456587289323943979621436184004),
	FLIT( 1.71499511953006020388132926947477070),
	FLIT(-0.95604710265072174471081704873906039),
	FLIT(-1.35887523049983854057602208066811761),
	FLIT( 3.98795959518091180465621612694578000),
	FLIT( 2.31127358524228647446536086042122559),
	FLIT( 0.79278272182537218311316090155088045),
	FLIT( 0.75215792450940400156561722375910192e-1),
	};

static const FTYPE c16[] = {
	FLIT( 8.27127622820435528072788337019492754),
	FLIT( 3.25831071437291242038314239477089362),
	FLIT( 0.99449813112099972756938862675587250),
	};

static const FTYPE s17[] = {	/* 31/2, 6.15 <= x < 10.123 */
	FLIT(-2388956.60240965685533531470422251877),
	FLIT( 11263279.7700574111244230863639733576),
	FLIT(-7952493.17753915133273600626307898819),
	FLIT( 1860982.53350701854176357779801513580),
	FLIT( 740022.239886813262986788251868569801),
	FLIT(-746513.089212838708462231845298508874),
	FLIT( 218497.240383790484235502098062287975),
	FLIT( 38680.8143843163058608598228982786238),
	FLIT(-59577.4649515947804728029733059384153),
	FLIT( 20133.9442992636328030766966059936462),
	FLIT( 2335.94014973713089800112525954035493),
	FLIT(-5072.38700374301855589751420580745427),
	FLIT( 1842.32896057890365740035664972415011),
	FLIT( 229.109623820488176282773912275707499),
	FLIT(-510.258388256696003379153450771429667),
	FLIT( 182.683246646661976375615472715832078),
	FLIT( 38.5946699701661258609988837160484241),
	FLIT(-64.8576433289163056555750880809991992),
	FLIT( 20.1580685469573436369046694035794814),
	FLIT( 9.39287353833084753530458432622321316),
	FLIT(-11.0265462511194162228820088531688057),
	FLIT( 2.19457311428953104534985871538006047),
	FLIT( 3.21245488636429725496898396433677263),
	FLIT(-2.69136211845851452760266190344729977),
	FLIT(-0.19287477627039812538409566393739270),
	FLIT( 1.84871729209434615287562918889923339),
	FLIT(-1.06306850190206200152357031184007331),
	FLIT(-1.43082218200037691958619343332793975),
	FLIT( 4.25700162480057190140419468138442320),
	FLIT( 2.41018101684941110868037909676722230),
	FLIT( 0.79646140705694881424826819144554721),
	FLIT( 0.72795389690102219128620146461154237e-1),
	};

static const FTYPE c17[] = {
	FLIT( 8.76190347525527217496663034206416922),
	FLIT( 3.39980076744381522485897123357204057),
	FLIT( 0.99553956821049210134428393212257917),
	};

static const FTYPE s18[] = {	/* 30/1, 10.123 <= x < 15 */
	FLIT(-8572179136.98391546923391054662074338),
	FLIT( 2154910053.59640413714129439094252161),
	FLIT( 284698435.679256432854757567671455553),
	FLIT(-283343241.407656670294664829506086521),
	FLIT( 43578545.6083893857955273953357660438),
	FLIT( 18452733.4890790028849018652932407791),
	FLIT(-9460885.64340701298006399078596914860),
	FLIT( 365457.728962033610930185515947214308),
	FLIT( 1011254.13916493094511623890677480314),
	FLIT(-307236.202344382577888652066777473037),
	FLIT(-48549.6215096402547777250277525302883),
	FLIT( 53688.3871548442187687770436562584413),
	FLIT(-7342.04080962074097252724894976451978),
	FLIT(-5984.42700118232273845807727403370148),
	FLIT( 2742.79096362941748543157201179327576),
	FLIT( 258.033397022912388916398423521422581),
	FLIT(-556.597675115442484690598520614888237),
	FLIT( 102.726773598704512240315664567307142),
	FLIT( 86.5063037849364336554072945336365672),
	FLIT(-47.2716352984332932362765925493653995),
	FLIT(-8.05834565685456866713484151082132230),
	FLIT( 15.3962556907573991897732046589850729),
	FLIT(-2.01344464091703060635214752416788579),
	FLIT(-4.96734363125488649729017476278631082),
	FLIT( 2.43261688639933141469139531728402729),
	FLIT( 1.84163797637129517816414218987829965),
	FLIT(-2.38149137531022955305940084541604759),
	FLIT(-0.92894260056711086802366513048395046),
	FLIT( 5.49414288888142365092374444277117220),
	FLIT( 1.02171963180584676015471405934846731),
	FLIT( 0.46514945174581365387071362290438040e-1),
	};

static const FTYPE c18[] = {
	FLIT( 9.95862423667610567166820875196450834),
	FLIT( 0.99999999999999999999999999999999959),
	};

static const FTYPE s19[] = {	/* 32/1, 15.0 <= x < 50.0 */
	FLIT(-256471837276.406463703375948684833272),
	FLIT( 34862981428.8814593607325529581829645),
	FLIT( 15486118850.6097691497529278530244763),
	FLIT(-4828524961.70899485820074070755902963),
	FLIT(-558562315.398866684122588926954574699),
	FLIT( 475453388.646778473039536674917025777),
	FLIT(-19216129.5825519019015177610420625866),
	FLIT(-39437096.8060042328549471539046068185),
	FLIT( 7369334.90226930043365679472393376915),
	FLIT( 2743462.74522619781370807242888760868),
	FLIT(-1134156.56935519943319256871519303397),
	FLIT(-125279.335477436658346832322009617334),
	FLIT( 143535.867142830215837803560900996333),
	FLIT(-6400.96335844482655870558998848441025),
	FLIT(-16693.7717299680448775432531396581653),
	FLIT( 3313.78214915713772900174151730035234),
	FLIT( 1815.03135737835651122257982343629885),
	FLIT(-765.763125951000830876788424112041846),
	FLIT(-170.872987904579197701165220036611981),
	FLIT( 158.127481145312834794821087289808623),
	FLIT( 6.86505111481973822919292750840129979),
	FLIT(-33.5152012505331141130962105677988558),
	FLIT( 3.98686322194144639171688150693618109),
	FLIT( 7.85885857325700865774398108767981763),
	FLIT(-2.54424465162471438768573700344411507),
	FLIT(-2.18438327915704456382113479818373032),
	FLIT( 1.45689101926916528096455866935362255),
	FLIT( 0.78929255542988455648581622233264949),
	FLIT(-1.16699186036105557757583077078484154),
	FLIT(-0.44061411803894108860608378649719852),
	FLIT( 2.53946505874062188105143889999961948),
	FLIT( 0.67443471315668294439558258251829813),
	FLIT( 0.24425325482056549603163711583736083e-1),
	};

static const FTYPE c19[] = {
	FLIT( 4.57831574047439538599830451239232451),
	FLIT( 0.99999999999999999999999999999999979),
	};
static const FTYPE s20[] = {	/* 31/2, 50.0 <= x < 107.0 */
	FLIT(-2424485616.63454425453403408066072411),
	FLIT( 1842165606.56112884562158688432621812),
	FLIT( 190005729.332568384555604636585019560),
	FLIT(-291625113.432526921108423875822602010),
	FLIT(-2179480.71341872186234321090095214460),
	FLIT( 36406201.7563882562695510500386201857),
	FLIT(-2087585.83205007910250565688366301223),
	FLIT(-4290739.76955026323377992019204050129),
	FLIT( 512845.324640014316495488567445396279),
	FLIT( 508392.613141132296477030335534572893),
	FLIT(-95541.4211472224865505731654537902725),
	FLIT(-62652.4360821517104287736240932359178),
	FLIT( 16892.3097228310963701097172347577951),
	FLIT( 8227.76278016379186943918871873802191),
	FLIT(-3074.27381975154751724234947819557816),
	FLIT(-1176.15838294440325566307757719577557),
	FLIT( 604.925476066079049407001848733235535),
	FLIT( 187.049015701363954378599755741530750),
	FLIT(-134.152981871040439746063482646267737),
	FLIT(-33.9392877363586465517729070836649257),
	FLIT( 35.0664575739905464319572345410099685),
	FLIT( 7.25711699352642150549065932074554760),
	FLIT(-11.4538780142302721646607410945585884),
	FLIT(-1.91512405368129065476115132483164386),
	FLIT( 5.12040910172118783326228316415766980),
	FLIT( 0.67264194391096064142628474631716758),
	FLIT(-3.74718595732217725560191772541469670),
	FLIT(-0.36686219915613271800427216058154297),
	FLIT( 8.11260815478732874402603336998552506),
	FLIT( 0.53376295397107164609544413404120629),
	FLIT( 0.57020113008203268521057936765894896),
	FLIT( 0.84603160592064241370607297313807371e-2),
	};

static const FTYPE c20[] = {
	FLIT( 14.8992232928842288699174324185119651),
	FLIT( 0.74532892659215347673523776524796237),
	FLIT( 0.99981375970883894713912603209476858),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.75), FLIT(3188141378479409216430037423419449.0)
	/ FLIT(5192296858534827628530496329220096.0), POLY(s10), POLY(c10)},
	{FLIT(2.15), FLIT(10764097141945954950840630263898139.0)
	/ FLIT(20769187434139310514121985316880384.0), POLY(s11), POLY(c11)},
	{FLIT(2.8), FLIT(2134706764929185777510461530700953.0)
	/ FLIT(5192296858534827628530496329220096.0), POLY(s12), POLY(c12)},
	{FLIT(3.5), FLIT(1668952561671908880599088105820745.0)
	/ FLIT(5192296858534827628530496329220096.0), POLY(s13), POLY(c13)},
	{FLIT(4.4), FLIT(2663580856001632354895514350703815.0)
	/ FLIT(10384593717069655257060992658440192.0), POLY(s14), POLY(c14)},
	{FLIT(5.4), FLIT(4283207846266103599292833672252267.0)
	/ FLIT(20769187434139310514121985316880384.0), POLY(s15), POLY(c15)},
	{FLIT(6.15), FLIT(2269267.0) 
	/ FLIT(16777216.0), POLY(s16), POLY(c16)},
	{FLIT(10.123), FLIT(2714393344912446822749346188720963.0) 
	/ FLIT(20769187434139310514121985316880384.0), POLY(s17), POLY(c17)},
	{FLIT(15), FLIT(1718147768803325424104470470268977.0)
	/ FLIT(20769187434139310514121985316880384.0), POLY(s18), POLY(c18)},
	{FLIT(50), FLIT(899998122146036788945286030398149.0)
	/ FLIT(20769187434139310514121985316880384.0), POLY(s19), POLY(c19)},
	{FLIT(106.910), FLIT(311537811512089657711829779753205.0)
	/ FLIT(20769187434139310514121985316880384.0), POLY(s20), POLY(c20)},
	};

 #else /* FBITS */
  #error erfc has insufficient precision
 #endif /* FBITS */

#if defined(_BUILD_FOR_RENESAS) && defined(_R_FLT)	/* V10000K102 */
static const FTYPE erfc_half = FLIT(4.7693627620446988e-01);
#else
static const FTYPE erfc_half = FLIT(0.47693627620446987338141835364313056);
#endif												/* V10000K102 */

FTYPE (FFUN(erfc))(FTYPE x)
	{	/* compute erfc(x) */
	switch (FNAME(Dtest)(&x))
		{	/* test for special codes */
	case _NANCODE:
		return (x);

	case 0:
		return (FLIT(1.0));

	case _INFCODE:
		return (FISNEG(x) ? FLIT(2.0) : FLIT(0.0));

	default:	/* finite */
		if (x < erfc_half)
			return (FLIT(1.0) - FFUN(erf)(x));
		else if (x < FLIT(1.523))
			if (x < mid0)
				{	/* compute approximation for [0.47, mid0) */
				x -= mid00;
				return (NPOLY00(x) / DPOLY00(x));
				}
			else
				{	/* compute approximation for [mid0, 1.523) */
				x -= mid01;
				return (NPOLY01(x) / DPOLY01(x));
				}
		else
			{	/* compute approximation for possibly finite tail */
			int i;

			for (i = 0; i < sizeof (approx) / sizeof (approx[0]); ++i)
				if (x < approx[i].to)
					{	/* found the interval, compute approximation */
					FTYPE w, y;

					y = FLIT(1.0) / x - approx[i].mid;
					w = FNAME(Poly)(y, approx[i].num, approx[i].nsize)
						/ FNAME(Poly)(y, approx[i].den, approx[i].dsize);

					y = x;	/* compute e-x*x in pieces */
					FNAME(Dint)(&y, FBITS / 3);
					w *= FFUN(exp)((y - x) * (y + x));
					y = - y * y;
					FNAME(Exp)(&y, w, 0);
					return (y);
					}

			_Feraise(_FE_UNDERFLOW);	/* definite underflow on tail */
			return (FLIT(0.0));
			}
		}
	}
_STD_END

/*
 * Copyright (c) 1992-2007 by P.J. Plauger.  ALL RIGHTS RESERVED.
 * Consult your license regarding permissions and restrictions.
V5.03:0216 */
