/* xxlgamma.h -- common lgamma[fl] functionality */
#include "xmath.h"
_STD_BEGIN

FTYPE FFUN(log1p)(FTYPE x);

FTYPE FNAME(Tgamma)(FTYPE *px, short *pex);
extern FTYPE FNAME(Gamma_big);

struct Approx {	/* description of one interval approximation */
	FTYPE to, mid;
	const FTYPE *num;
	int nsize;
	const FTYPE *den;
	int dsize;
	};

struct RootApprox {	/* description of one interval root approximation */
	const FTYPE from, to;
	const FTYPE zero1, zero2;
	const FTYPE *sr;
	int srsize;
	};

 #define ASIZE(x)	(sizeof (x) / sizeof ((x)[0]))
 #define PSIZE(x)	(ASIZE(x) - 1)
 #define ROOTPOLY(x)	x, PSIZE(x)
 #define POLY1	ROOTPOLY
 #define NROOTS		(sizeof (rootapprox) / sizeof (rootapprox[0]))

 #if   FBITS <= 30
  #define DECLARE(w, xinv)
  #define POLY(xinv, w)	(c[0] * xinv)

static const FTYPE c[] = {	/* Stirling fixup order 0, Gamma_big < x */
	FLIT( 0.0833333333),
	};

/* Polynomial approximation fixup for 1.2 - 1.5
 */
static const FTYPE s10[] = {	/* 7/1, 1.2 <= x < 1.5 */
	FLIT( 0.5360760204e-3),
	FLIT(-0.2213148498e-2),
	FLIT( 0.7719251053e-2),
	FLIT(-0.2947412669e-1),
	FLIT( 0.1649097161),
	FLIT( 0.4654077568),
	FLIT(-0.1879605312),
	FLIT(-0.1152409017),
	};

static const FTYPE c10[] = {
	FLIT( 0.6424155168),
	FLIT( 1.0000000153),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.5), FLIT(11324621.0) / FLIT(8388608.0), POLY1(s10), POLY1(c10)},
	};

/* Taylor expansions for negative roots
 */
static const FTYPE sr00[] = {	/* 5/0, fn(x-x0), -7.00023 < x <= -7.00017 */
	FLIT( .1615684343e15),
	FLIT( .4272589080e11),
	FLIT( 12710958.83),
	FLIT( 5044.029941),
	FLIT(-0.5743202284e-11),
	};

static const FTYPE sr01[] = {	/* 4/0, fn(x-x0), -6.99988 < x <= -6.99976 */
	FLIT(-.4262348976e11),
	FLIT( 12690641.17),
	FLIT(-5035.967374),
	FLIT( 0.5313011066e-13),
	};

static const FTYPE sr02[] = {	/* 11/0, fn(x-x0), -6.00176 < x <= -6.00137 */
	FLIT( .3842191857e28),
	FLIT( .5913963410e25),
	FLIT( .9216653296e22),
	FLIT( .1459174708e20),
	FLIT( .2358284399e17),
	FLIT( .3920301958e14),
	FLIT( .6788465695e11),
	FLIT( 125386984.9),
	FLIT( 260548.4103),
	FLIT( 723.7366299),
	FLIT( 0.8310060495e-11),
	};

static const FTYPE sr03[] = {	/* 3/0, fn(x-x0), -5.9987 < x <= -5.9985 */
	FLIT( 257851.5896),
	FLIT(-716.2454305),
	FLIT( 0.1671250568e-11),
	};

static const FTYPE sr04[] = {	/* 10/0, fn(x-x0), -5.01 < x <= -5.0064 */
	FLIT( .6498037322e18),
	FLIT( .6007721003e16),
	FLIT( .5642567136e14),
	FLIT( .5410016092e12),
	FLIT( 5335250743.),
	FLIT( 54807486.36),
	FLIT( 600556.1685),
	FLIT( 7404.762432),
	FLIT( 123.3621846),
	FLIT(-0.7615545600e-12),
	};

static const FTYPE sr05[] = {	/* 9/0, fn(x-x0), -4.99266 < x <= -4.991 */
	FLIT( .4784685740e16),
	FLIT(-.4623570027e14),
	FLIT( .4560960423e12),
	FLIT(-4627747172.),
	FLIT( 48911582.71),
	FLIT(-551419.9979),
	FLIT( 6995.235955),
	FLIT(-116.5357816),
	FLIT( 0.3281973185e-12),
	};

static const FTYPE sr06[] = {	/* 23/0, fn(x-x0), -4.062 < x <= -4.0255 */
	FLIT( .3680601218e30),
	FLIT( .1517740560e29),
	FLIT( .6272811371e27),
	FLIT( .2599046273e26),
	FLIT( .1079867565e25),
	FLIT( .4500590188e23),
	FLIT( .1882234790e22),
	FLIT( .7902743912e20),
	FLIT( .3332855779e19),
	FLIT( .1412786684e18),
	FLIT( .6024404002e16),
	FLIT( .2586890453e15),
	FLIT( .1120072441e14),
	FLIT( .4898679104e12),
	FLIT( .2169236496e11),
	FLIT( 975830163.5),
	FLIT( 44812215.95),
	FLIT( 2116669.418),
	FLIT( 104145.5552),
	FLIT( 5465.691821),
	FLIT( 324.2553229),
	FLIT( 26.79048089),
	FLIT( 0.1813014890e-13),
	};

static const FTYPE sr07[] = {	/* 10/0, fn(x-x0), -3.965 < x <= -3.953 */
	FLIT(-.1557722425e12),
	FLIT( 7834398184.),
	FLIT(-400276998.2),
	FLIT( 20877114.73),
	FLIT(-1119991.502),
	FLIT( 62588.08192),
	FLIT(-3730.604716),
	FLIT( 251.7146826),
	FLIT(-20.72506084),
	FLIT(-0.4643326223e-13),
	};

static const FTYPE sr08[] = {	/* 16/0, fn(x-x0), -3.1962 < x <= -3.121 */
	FLIT( .2933993859e12),
	FLIT( .4513558336e11),
	FLIT( 6979115397.),
	FLIT( 1085573222.),
	FLIT( 170037345.8),
	FLIT( 26855525.06),
	FLIT( 4284377.444),
	FLIT( 692049.6023),
	FLIT( 113559.6642),
	FLIT( 19023.06198),
	FLIT( 3277.193785),
	FLIT( 588.8907421),
	FLIT( 112.2689863),
	FLIT( 25.83133838),
	FLIT( 7.781884657),
	FLIT(-0.1757985417e-14),
	};

static const FTYPE sr09[] = {	/* 19/0, fn(x-x0), -2.864 < x <= -2.65 */
	FLIT( 3233548126.),
	FLIT(-863873252.9),
	FLIT( 231593359.8),
	FLIT(-62330684.83),
	FLIT( 16850484.27),
	FLIT(-4578716.595),
	FLIT( 1251567.447),
	FLIT(-344497.5934),
	FLIT( 95617.45164),
	FLIT(-26804.57842),
	FLIT( 7610.397987),
	FLIT(-2193.112776),
	FLIT( 646.9059998),
	FLIT(-194.7661553),
	FLIT( 62.62728272),
	FLIT(-20.09513491),
	FLIT( 9.575189476),
	FLIT(-1.914350186),
	FLIT(-0.6767913369e-15),
	};

static const FTYPE sr10[] = {	/* 18/0, fn(x-x0), -2.65 < x <= -2.265 */
	FLIT( 33630.45706),
	FLIT( 18347.17070),
	FLIT( 7776.158102),
	FLIT( 4487.218453),
	FLIT( 1811.205419),
	FLIT( 1130.457858),
	FLIT( 425.1798458),
	FLIT( 296.4364134),
	FLIT( 100.6481234),
	FLIT( 82.22915683),
	FLIT( 24.03882306),
	FLIT( 24.82494211),
	FLIT( 5.800414568),
	FLIT( 8.721782585),
	FLIT( 1.411291143),
	FLIT( 4.858320951),
	FLIT( 1.515603447),
	FLIT(-0.2636060628e-14),
	};

static const FTYPE sr11[] = {	/* 4/0, fn(x-x0), 2.72 < x <= 2.77 */
	FLIT(-0.3172182974e-1),
	FLIT( 0.2197332343),
	FLIT( 0.8145158497),
	FLIT( 0.4670475668),
	};

static const struct RootApprox rootapprox[] = {	/* table of roots */
	{FLIT(-7.00023), FLIT(-7.00017),
	FLIT(-14680480.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(1085455.0) / FLIT(16777216.0) / FLIT(2097152.0),
	ROOTPOLY(sr00)},

	{FLIT(-6.99988), FLIT(-6.99976),
	FLIT(-14679648.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(8996753.0) / FLIT(16777216.0) / FLIT(4194304.0),
	ROOTPOLY(sr01)},

	{FLIT(-6.00176), FLIT(-6.00137),
	FLIT(-12585818.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(3468551.0) / FLIT(16777216.0) / FLIT(524288.0),
	ROOTPOLY(sr02)},

	{FLIT(-5.9987), FLIT(-5.9985),
	FLIT(-12579992.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(10936511.0) / FLIT(16777216.0) / FLIT(4194304.0),
	ROOTPOLY(sr03)},

	{FLIT(-5.01), FLIT(-5.0064),
	FLIT(-10502995.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(4225609.0) / FLIT(16777216.0) / FLIT(2097152.0),
	ROOTPOLY(sr04)},

	{FLIT(-4.99266), FLIT(-4.991),
	FLIT(-10468028.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(364821.0) / FLIT(16777216.0) / FLIT(262144.0),
	ROOTPOLY(sr05)},

	{FLIT(-4.062), FLIT(-4.0255),
	FLIT(-8471156.0) / FLIT(16777216.0) * FLIT(8.0),
	FLIT(1002709.0) / FLIT(16777216.0) / FLIT(524288.0),
	ROOTPOLY(sr06)},

	{FLIT(-3.965), FLIT(-3.953),
	FLIT(-16589707.0) / FLIT(16777216.0) * FLIT(4.0),
	FLIT(6037121.0) / FLIT(16777216.0) / FLIT(4194304.0),
	ROOTPOLY(sr07)},

	{FLIT(-3.1962), FLIT(-3.121),
	FLIT(-13185134.0) / FLIT(16777216.0) * FLIT(4.0),
	FLIT(3545701.0) / FLIT(16777216.0) / FLIT(8388608.0),
	ROOTPOLY(sr08)},

	{FLIT(-2.864), FLIT(-2.65),
	FLIT(-11524617.0) / FLIT(16777216.0) * FLIT(4.0),
	FLIT(1434663.0) / FLIT(16777216.0) / FLIT(524288.0),
	ROOTPOLY(sr09)},

	{FLIT(-2.65), FLIT(-2.265),
	FLIT(-10305509.0) / FLIT(16777216.0) * FLIT(4.0),
	FLIT(10481785.0) / FLIT(16777216.0) / FLIT(8388608.0),
	ROOTPOLY(sr10)},

	{FLIT(2.72), FLIT(2.77),
	FLIT(11492392.0) / FLIT(16777216.0) * FLIT(4.0),
	FLIT(16106127.0) / FLIT(16777216.0) / FLIT(4194304.0),
	ROOTPOLY(sr11)},
	};

 #elif FBITS <= 53
  #define DECLARE(w, xinv)	FTYPE w = xinv * xinv;
  #define POLY(xinv, w)	(((c[0] * w + c[1]) * w + c[2]) * xinv)

#if defined(_BUILD_FOR_RENESAS) && defined(_R_FLT)	/* V10000K102 */
static const FTYPE c[] = {	/* Stirling fixup order 2, Gamma_big < x */
	FLIT( 7.9365079365079365e-04),
	FLIT(-2.7777777777777779e-03),
	FLIT( 8.3333333333333329e-02),
	};

/* Polynomial approximation fixup for 1.2 - 1.5
 */
static const FTYPE s10[] = {	/* 12/2, 1.2 <= x < 1.5 */
	FLIT( 6.6454129398121538e-07),
	FLIT(-3.7089172292679655e-06),
	FLIT( 1.4370364744985201e-05),
	FLIT(-4.9032574258831847e-05),
	FLIT( 1.6172652190598458e-04),
	FLIT(-5.4911719631636627e-04),
	FLIT( 2.0525849632769453e-03),
	FLIT(-9.4072115542101469e-03),
	FLIT( 6.8944965881171286e-02),
	FLIT( 4.5072592886235285e-01),
	FLIT( 3.4855136383920748e-01),
	FLIT(-2.5793049510317639e-01),
	FLIT(-1.1473652220196856e-01),
	};

static const FTYPE c10[] = {
	FLIT( 3.8904024299563883e-01),
	FLIT( 1.2539047749924295e+00),
	FLIT( 9.9562329726629917e-01),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.5), FLIT( 3.0399297484750850e+15)
		/ FLIT( 2.2517998136852480e+15), POLY1(s10), POLY1(c10)},
	};

/* Taylor expansions for negative roots
 */
static const FTYPE sr00[] = {	/* 5/0, fn(x-x0), -7.00023 < x <= -7.00017 */
	FLIT( 1.6156843435328284e+14),
	FLIT( 4.2725890801879196e+10),
	FLIT( 1.2710958833951393e+07),
	FLIT( 5.0440299411108290e+03),
	FLIT(-1.2175258490193363e-28),
	};

static const FTYPE sr01[] = {	/* 5/0, fn(x-x0), -6.99988 < x <= -6.99976 */
	FLIT( 1.6105233314447378e+14),
	FLIT(-4.2623489764758118e+10),
	FLIT( 1.2690641166047178e+07),
	FLIT(-5.0359673737681251e+03),
	FLIT(-2.0581818739894355e-30),
	};

static const FTYPE sr02[] = {	/* 25/0, fn(x-x0), -6.00176 < x <= -6.00137 */
	FLIT( 1.6702384292615088e+67),
	FLIT( 2.4143708154808178e+64),
	FLIT( 3.4966424302971770e+61),
	FLIT( 5.0745402854220778e+58),
	FLIT( 7.3812191351926530e+55),
	FLIT( 1.0763328342636470e+53),
	FLIT( 1.5738733442238089e+50),
	FLIT( 2.3085296615879001e+47),
	FLIT( 3.3978679185880896e+44),
	FLIT( 5.0208507521853192e+41),
	FLIT( 7.4521678897734851e+38),
	FLIT( 1.1117558137197870e+36),
	FLIT( 1.6684515921763653e+33),
	FLIT( 2.5214146210905919e+30),
	FLIT( 3.8421918575710541e+27),
	FLIT( 5.9139634091676227e+24),
	FLIT( 9.2166532952423808e+21),
	FLIT( 1.4591747070346195e+19),
	FLIT( 2.3582843991458176e+16),
	FLIT( 3.9203019565186828e+13),
	FLIT( 6.7884656938770958e+10),
	FLIT( 1.2538698494090366e+08),
	FLIT( 2.6054841030309396e+05),
	FLIT( 7.2373662992528011e+02),
	FLIT( 8.0774867792891732e-31),
	};   

static const FTYPE sr03[] = {	/* 5/0, fn(x-x0), -5.9987 < x <= -5.9985 */
	FLIT( 6.6486633581993340e+10),
	FLIT(-1.2344528215783688e+08),
	FLIT( 2.5785158963956262e+05),
	FLIT(-7.1624543042754726e+02),
	FLIT( 2.4868202361581571e-30),
	};

static const FTYPE sr04[] = {	/* 22/0, fn(x-x0), -5.00992 < x <= -5.0066 */
	FLIT( 2.9343034918502209e+42),
	FLIT( 2.5320330005878998e+40),
	FLIT( 2.1903866733887265e+38),
	FLIT( 1.9001020060889469e+36),
	FLIT( 1.6533908593556386e+34),
	FLIT( 1.4437084659298575e+32),
	FLIT( 1.2655615127215833e+30),
	FLIT( 1.1143497358009344e+28),
	FLIT( 9.8623685989765991e+25),
	FLIT( 8.7804822306338076e+23),
	FLIT( 7.8719433736243705e+21),
	FLIT( 7.1162251237406753e+19),
	FLIT( 6.4980373209299763e+17),
	FLIT( 6.0077210028634350e+15),
	FLIT( 5.6425671356242344e+13),
	FLIT( 5.4100160907477380e+11),
	FLIT( 5.3352507433843842e+09),
	FLIT( 5.4807486356655858e+07),
	FLIT( 6.0055616845281096e+05),
	FLIT( 7.4047624322286820e+03),
	FLIT( 1.2336218456335340e+02),
	FLIT(-4.2308313888395784e-30),
	};   

static const FTYPE sr05[] = {	/* 13/0, fn(x-x0), -4.9916 < x <= -4.991 */
	FLIT( 6.2407079930079154e+23),
	FLIT(-5.7564468262543766e+21),
	FLIT( 5.3540109714248598e+19),
	FLIT(-5.0300096898716947e+17),
	FLIT( 4.7846857403596570e+15),
	FLIT(-4.6235700276520070e+13),
	FLIT( 4.5609604226219757e+11),
	FLIT(-4.6277471717392492e+09),
	FLIT( 4.8911582709059045e+07),
	FLIT(-5.5141999785128841e+05),
	FLIT( 6.9952359548940640e+03),
	FLIT(-1.1653578161624363e+02),
	FLIT(-1.1238146637063361e-31),
	};

static const FTYPE sr06[] = {	/* 25/0, fn(x-x0), -4.05 < x <= -4.0338 */
	FLIT( 2.1776066140073501e+32),
	FLIT( 8.9441324408859603e+30),
	FLIT( 3.6806012174521232e+29),
	FLIT( 1.5177405599732661e+28),
	FLIT( 6.2728113723849414e+26),
	FLIT( 2.5990462732887710e+25),
	FLIT( 1.0798675638117844e+24),
	FLIT( 4.5005901869396571e+22),
	FLIT( 1.8822347903091118e+21),
	FLIT( 7.9027439114890969e+19),
	FLIT( 3.3328557787986145e+18),
	FLIT( 1.4127866850826970e+17),
	FLIT( 6.0244040009605240e+15),
	FLIT( 2.5868904525606241e+14),
	FLIT( 1.1200724416202004e+13),
	FLIT( 4.8986791049820020e+11),
	FLIT( 2.1692364960713310e+10),
	FLIT( 9.7583016353821456e+08),
	FLIT( 4.4812215950155713e+07),
	FLIT( 2.1166694175775349e+06),
	FLIT( 1.0414555520145541e+05),
	FLIT( 5.4656918207771341e+03),
	FLIT( 3.2425532293784715e+02),
	FLIT( 2.6790480886140593e+01),
	FLIT( 9.6342376492948458e-32),
	};

static const FTYPE sr07[] = {	/* 27/0, fn(x-x0), -3.957 < x <= -3.943 */
	FLIT( 4.7379169342948859e+33),
	FLIT(-2.2028444342134063e+32),
	FLIT( 1.0258305810079953e+31),
	FLIT(-4.7854424074312094e+29),
	FLIT( 2.2366101714185069e+28),
	FLIT(-1.0475065040607433e+27),
	FLIT( 4.9171003748270798e+25),
	FLIT(-2.3139209334600083e+24),
	FLIT( 1.0919246178382312e+23),
	FLIT(-5.1686757445722649e+21),
	FLIT( 2.4551117958856704e+20),
	FLIT(-1.1707469715963822e+19),
	FLIT( 5.6077586373050368e+17),
	FLIT(-2.6998338793041112e+16),
	FLIT( 1.3075617136538245e+15),
	FLIT(-6.3769616182266906e+13),
	FLIT( 3.1359529252932925e+12),
	FLIT(-1.5577224241657217e+11),
	FLIT( 7.8343981837383184e+09),
	FLIT(-4.0027699818746364e+08),
	FLIT( 2.0877114726503327e+07),
	FLIT(-1.1199915016655221e+06),
	FLIT( 6.2588081918766060e+04),
	FLIT(-3.7306047156806126e+03),
	FLIT( 2.5171468258688941e+02),
	FLIT(-2.0725060845803707e+01),
	FLIT(-4.3412455472202367e-31),
	};

static const FTYPE sr08[] = {	/* 23/0, fn(x-x0), -3.177 < x <= -3.121 */
	FLIT( 1.5901984525424195e+17),
	FLIT( 2.3919458772922344e+16),
	FLIT( 3.6060959964402615e+15),
	FLIT( 5.4501733330968225e+14),
	FLIT( 8.2601521376228047e+13),
	FLIT( 1.2557646866390602e+13),
	FLIT( 1.9157279734539404e+12),
	FLIT( 2.9339938588233044e+11),
	FLIT( 4.5135583357515282e+10),
	FLIT( 6.9791153966713457e+09),
	FLIT( 1.0855732215700052e+09),
	FLIT( 1.7003734582174313e+08),
	FLIT( 2.6855525055575889e+07),
	FLIT( 4.2843774436459793e+06),
	FLIT( 6.9204960228977457e+05),
	FLIT( 1.1355966423713516e+05),
	FLIT( 1.9023061984356358e+04),
	FLIT( 3.2771937854953417e+03),
	FLIT( 5.8889074223800139e+02),
	FLIT( 1.1226898629717600e+02),
	FLIT( 2.5831338372387957e+01),
	FLIT( 7.7818846581313510e+00),
	FLIT(-1.0431866962787153e-31),
	};

static const FTYPE sr09[] = {	/* 32/0, fn(x-x0), -2.821 < x <= -2.66 */
	FLIT(-1.1175773048045936e+17),
	FLIT( 2.9138361921333636e+16),
	FLIT(-7.6056355434718370e+15),
	FLIT( 1.9875707527950640e+15),
	FLIT(-5.2007261370517388e+14),
	FLIT( 1.3627039714588525e+14),
	FLIT(-3.5758721374667469e+13),
	FLIT( 9.3984853476306484e+12),
	FLIT(-2.4745053366684849e+12),
	FLIT( 6.5274066620752576e+11),
	FLIT(-1.7254054949328647e+11),
	FLIT( 4.5711723541795547e+10),
	FLIT(-1.2140906400864338e+10),
	FLIT( 3.2335481260760565e+09),
	FLIT(-8.6387325292932093e+08),
	FLIT( 2.3159335980352938e+08),
	FLIT(-6.2330684832407415e+07),
	FLIT( 1.6850484270697366e+07),
	FLIT(-4.5787165945817176e+06),
	FLIT( 1.2515674465251002e+06),
	FLIT(-3.4449759343483287e+05),
	FLIT( 9.5617451643349705e+04),
	FLIT(-2.6804578420618109e+04),
	FLIT( 7.6103979866583286e+03),
	FLIT(-2.1931127764806238e+03),
	FLIT( 6.4690599971292829e+02),
	FLIT(-1.9476615530344623e+02),
	FLIT( 6.2627282713513715e+01),
	FLIT(-2.0095134916842603e+01),
	FLIT( 9.5751894757096672e+00),
	FLIT(-1.9143501856115988e+00),
	FLIT( 6.3609282465313464e-33),
	};   

static const FTYPE sr10[] = {	/* 42/0, fn(x-x0), -2.66 < x <= -2.28 */
	FLIT( 2.1346022881517327e+12),
	FLIT( 1.0018257054778754e+12),
	FLIT( 4.6855743010287335e+11),
	FLIT( 2.2035815144677905e+11),
	FLIT( 1.0310728438127779e+11),
	FLIT( 4.8612189337342163e+10),
	FLIT( 2.2750853238533039e+10),
	FLIT( 1.0759909800736023e+10),
	FLIT( 5.0350140002995262e+09),
	FLIT( 2.3907009360723963e+09),
	FLIT( 1.1179539850661736e+09),
	FLIT( 5.3351888554834527e+08),
	FLIT( 2.4911897461233804e+08),
	FLIT( 1.1967450386711416e+08),
	FLIT( 5.5731833611605622e+07),
	FLIT( 2.7007615310435627e+07),
	FLIT( 1.2522259656340322e+07),
	FLIT( 6.1393858152913544e+06),
	FLIT( 2.8270236968710567e+06),
	FLIT( 1.4079795933782964e+06),
	FLIT( 6.4156832774174784e+05),
	FLIT( 3.2643370222156332e+05),
	FLIT( 1.4643199829213394e+05),
	FLIT( 7.6721036023984285e+04),
	FLIT( 3.3630457055794337e+04),
	FLIT( 1.8347170699977025e+04),
	FLIT( 7.7761581023442468e+03),
	FLIT( 4.4872184531889980e+03),
	FLIT( 1.8112054188018712e+03),
	FLIT( 1.1304578581775129e+03),
	FLIT( 4.2517984568438214e+02),
	FLIT( 2.9643641350761322e+02),
	FLIT( 1.0064812354551160e+02),
	FLIT( 8.2229156827042530e+01),
	FLIT( 2.4038823062292956e+01),
	FLIT( 2.4824942121894072e+01),
	FLIT( 5.8004145665998728e+00),
	FLIT( 8.7217825838153455e+00),
	FLIT( 1.4112911430779800e+00),
	FLIT( 4.8583209516339965e+00),
	FLIT( 1.5156034480216574e+00),
	FLIT( 2.0646555198525409e-33),
	};

static const struct RootApprox rootapprox[] = {	/* table of roots */
	{FLIT(-7.00023), FLIT(-7.00017),
	FLIT(-7.8815226514631990e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 6.3492921427783900e+14) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 2.8147497671065600e+14),
	ROOTPOLY(sr00)},

	{FLIT(-6.99988), FLIT(-6.99976),
	FLIT(-7.8810758656509280e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 3.4237129406453790e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 9.0071992547409920e+15) / FLIT(4.0),
	ROOTPOLY(sr01)},

	{FLIT(-6.00176), FLIT(-6.00137),
	FLIT(-6.7569591439515010e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 2.6025768529667130e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 4.5035996273704960e+15),
	ROOTPOLY(sr02)},

	{FLIT(-5.9987), FLIT(-5.9985),
	FLIT(-6.7538316030085260e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 5.6485716873465350e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 1.1258999068426240e+15),
	ROOTPOLY(sr03)},

	{FLIT(-5.00992), FLIT(-5.0066),
	FLIT(-5.6387523691619460e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 8.5617331077912030e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 1.1258999068426240e+15),
	ROOTPOLY(sr04)},

	{FLIT(-4.9916), FLIT(-4.991),
	FLIT(-5.6199796458073570e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 6.1554726964476950e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 4.5035996273704960e+15),
	ROOTPOLY(sr05)},

	{FLIT(-4.05), FLIT(-4.0338),
	FLIT(-4.5479171190675210e+15) / FLIT( 9.0071992547409920e+15) * FLIT(8.0),
	FLIT( 4.2885117746950390e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 2.2517998136852480e+15),
	ROOTPOLY(sr06)},

	{FLIT(-3.957), FLIT(-3.943),
	FLIT(-8.9065309337149180e+15) / FLIT( 9.0071992547409920e+15) * FLIT(4.0),
	FLIT( 8.6016669985952930e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 2.2517998136852480e+15),
	ROOTPOLY(sr07)},

	{FLIT(-3.177), FLIT(-3.121),
	FLIT(-7.0787148586909920e+15) / FLIT( 9.0071992547409920e+15) * FLIT(4.0),
	FLIT( 2.2909733245148350e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 1.1258999068426240e+15),
	ROOTPOLY(sr08)},

	{FLIT(-2.821), FLIT(-2.66),
	FLIT(-6.1872312719669770e+15) / FLIT( 9.0071992547409920e+15) * FLIT(4.0),
	FLIT( 7.3465648664042290e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 9.0071992547409920e+15),
	ROOTPOLY(sr09)},

	{FLIT(-2.66), FLIT(-2.28),
	FLIT(-5.5327278477456450e+15) / FLIT( 9.0071992547409920e+15) * FLIT(4.0),
	FLIT( 4.1276082649375870e+15) / FLIT( 9.0071992547409920e+15)
		/ FLIT( 1.1258999068426240e+15),
	ROOTPOLY(sr10)},
	};
#else
static const FTYPE c[] = {	/* Stirling fixup order 2, Gamma_big < x */
	FLIT( 0.00079365079365079364),
	FLIT(-0.00277777777777777777),
	FLIT( 0.08333333333333333333),
	};

/* Polynomial approximation fixup for 1.2 - 1.5
 */
static const FTYPE s10[] = {	/* 12/2, 1.2 <= x < 1.5 */
	FLIT( 0.66454129398121535842e-6),
	FLIT(-0.37089172292679657345e-5),
	FLIT( 0.14370364744985202372e-4),
	FLIT(-0.49032574258831844388e-4),
	FLIT( 0.16172652190598458749e-3),
	FLIT(-0.54911719631636623561e-3),
	FLIT( 0.20525849632769452598e-2),
	FLIT(-0.94072115542101477245e-2),
	FLIT( 0.68944965881171283469e-1),
	FLIT( 0.45072592886235283741),
	FLIT( 0.34855136383920749813),
	FLIT(-0.25793049510317640659),
	FLIT(-0.11473652220196855901),
	};

static const FTYPE c10[] = {
	FLIT( 0.38904024299563881796),
	FLIT( 1.25390477499242941917),
	FLIT( 0.99562329726629917468),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.5), FLIT(3039929748475085.0)
		/ FLIT(2251799813685248.0), POLY1(s10), POLY1(c10)},
	};

/* Taylor expansions for negative roots
 */
static const FTYPE sr00[] = {	/* 5/0, fn(x-x0), -7.00023 < x <= -7.00017 */
	FLIT( 161568434353282.84645),
	FLIT( 42725890801.879194201),
	FLIT( 12710958.833951394096),
	FLIT( 5044.0299411108291964),
	FLIT(-0.12175258490193364201e-27),
	};

static const FTYPE sr01[] = {	/* 5/0, fn(x-x0), -6.99988 < x <= -6.99976 */
	FLIT( 161052333144473.79179),
	FLIT(-42623489764.758116460),
	FLIT( 12690641.166047179245),
	FLIT(-5035.9673737681254144),
	FLIT(-0.20581818739894356439e-29),
	};

static const FTYPE sr02[] = {	/* 25/0, fn(x-x0), -6.00176 < x <= -6.00137 */
	FLIT( .16702384292615087643e68),
	FLIT( .24143708154808177927e65),
	FLIT( .34966424302971768883e62),
	FLIT( .50745402854220778545e59),
	FLIT( .73812191351926529454e56),
	FLIT( .10763328342636469567e54),
	FLIT( .15738733442238088969e51),
	FLIT( .23085296615878999430e48),
	FLIT( .33978679185880898173e45),
	FLIT( .50208507521853188549e42),
	FLIT( .74521678897734857594e39),
	FLIT( .11117558137197869630e37),
	FLIT( .16684515921763651638e34),
	FLIT( .25214146210905917811e31),
	FLIT( .38421918575710539199e28),
	FLIT( .59139634091676227619e25),
	FLIT( .92166532952423812779e22),
	FLIT( 14591747070346194018.),
	FLIT( 23582843991458177.616),
	FLIT( 39203019565186.829430),
	FLIT( 67884656938.770961215),
	FLIT( 125386984.94090366666),
	FLIT( 260548.41030309396963),
	FLIT( 723.73662992528012397),
	FLIT( 0.80774867792891736990e-30),
	};

static const FTYPE sr03[] = {	/* 5/0, fn(x-x0), -5.9987 < x <= -5.9985 */
	FLIT( 66486633581.993339903),
	FLIT(-123445282.15783688645),
	FLIT( 257851.58963956261716),
	FLIT(-716.24543042754729207),
	FLIT( 0.24868202361581571484e-29),
	};

static const FTYPE sr04[] = {	/* 22/0, fn(x-x0), -5.00992 < x <= -5.0066 */
	FLIT( .29343034918502210617e43),
	FLIT( .25320330005878996474e41),
	FLIT( .21903866733887263598e39),
	FLIT( .19001020060889468230e37),
	FLIT( .16533908593556384527e35),
	FLIT( .14437084659298575148e33),
	FLIT( .12655615127215832952e31),
	FLIT( .11143497358009343437e29),
	FLIT( .98623685989765983922e26),
	FLIT( .87804822306338075083e24),
	FLIT( .78719433736243702136e22),
	FLIT( 71162251237406755435.),
	FLIT( 649803732092997587.50),
	FLIT( 6007721002863435.3097),
	FLIT( 56425671356242.343690),
	FLIT( 541001609074.77377555),
	FLIT( 5335250743.3843846461),
	FLIT( 54807486.356655854741),
	FLIT( 600556.16845281092210),
	FLIT( 7404.7624322286821683),
	FLIT( 123.36218456335339335),
	FLIT(-0.42308313888395786116e-29),
	};

static const FTYPE sr05[] = {	/* 13/0, fn(x-x0), -4.9916 < x <= -4.991 */
	FLIT( .62407079930079160892e24),
	FLIT(-.57564468262543762995e22),
	FLIT( 53540109714248596276.),
	FLIT(-503000968987169457.12),
	FLIT( 4784685740359657.1393),
	FLIT(-46235700276520.071504),
	FLIT( 456096042262.19758599),
	FLIT(-4627747171.7392495881),
	FLIT( 48911582.709059044464),
	FLIT(-551419.99785128835644),
	FLIT( 6995.2359548940638411),
	FLIT(-116.53578161624363080),
	FLIT(-0.11238146637063361009e-30),
	};

static const FTYPE sr06[] = {	/* 25/0, fn(x-x0), -4.05 < x <= -4.0338 */
	FLIT( .21776066140073500912e33),
	FLIT( .89441324408859604541e31),
	FLIT( .36806012174521234933e30),
	FLIT( .15177405599732660536e29),
	FLIT( .62728113723849412533e27),
	FLIT( .25990462732887708421e26),
	FLIT( .10798675638117843503e25),
	FLIT( .45005901869396575263e23),
	FLIT( .18822347903091117223e22),
	FLIT( 79027439114890969640.),
	FLIT( 3332855778798614561.3),
	FLIT( 141278668508269686.59),
	FLIT( 6024404000960524.0985),
	FLIT( 258689045256062.42226),
	FLIT( 11200724416202.002935),
	FLIT( 489867910498.20017235),
	FLIT( 21692364960.713309683),
	FLIT( 975830163.53821450990),
	FLIT( 44812215.950155709152),
	FLIT( 2116669.4175775350565),
	FLIT( 104145.55520145541151),
	FLIT( 5465.6918207771342500),
	FLIT( 324.25532293784715720),
	FLIT( 26.790480886140593262),
	FLIT( 0.96342376492948458900e-31),
	};

static const FTYPE sr07[] = {	/* 27/0, fn(x-x0), -3.957 < x <= -3.943 */
	FLIT( .47379169342948857155e34),
	FLIT(-.22028444342134064193e33),
	FLIT( .10258305810079954444e32),
	FLIT(-.47854424074312096897e30),
	FLIT( .22366101714185069414e29),
	FLIT(-.10475065040607433448e28),
	FLIT( .49171003748270800414e26),
	FLIT(-.23139209334600083497e25),
	FLIT( .10919246178382310542e24),
	FLIT(-.51686757445722645660e22),
	FLIT( .24551117958856703321e21),
	FLIT(-11707469715963823368.),
	FLIT( 560775863730503686.54),
	FLIT(-26998338793041111.225),
	FLIT( 1307561713653824.5426),
	FLIT(-63769616182266.903880),
	FLIT( 3135952925293.2922916),
	FLIT(-155772242416.57216815),
	FLIT( 7834398183.7383184021),
	FLIT(-400276998.18746365804),
	FLIT( 20877114.726503329247),
	FLIT(-1119991.5016655221355),
	FLIT( 62588.081918766060559),
	FLIT(-3730.6047156806125108),
	FLIT( 251.71468258688940014),
	FLIT(-20.725060845803705677),
	FLIT(-0.43412455472202368124e-30),
	};

static const FTYPE sr08[] = {	/* 23/0, fn(x-x0), -3.177 < x <= -3.121 */
	FLIT( 159019845254241963.38),
	FLIT( 23919458772922345.047),
	FLIT( 3606095996440261.2531),
	FLIT( 545017333309682.24398),
	FLIT( 82601521376228.045059),
	FLIT( 12557646866390.601851),
	FLIT( 1915727973453.9403584),
	FLIT( 293399385882.33045051),
	FLIT( 45135583357.515278640),
	FLIT( 6979115396.6713459663),
	FLIT( 1085573221.5700050825),
	FLIT( 170037345.82174312364),
	FLIT( 26855525.055575888049),
	FLIT( 4284377.4436459796216),
	FLIT( 692049.60228977461199),
	FLIT( 113559.66423713516225),
	FLIT( 19023.061984356358371),
	FLIT( 3277.1937854953417450),
	FLIT( 588.89074223800134673),
	FLIT( 112.26898629717600257),
	FLIT( 25.831338372387958936),
	FLIT( 7.7818846581313508722),
	FLIT(-0.10431866962787154032e-30),
	};

static const FTYPE sr09[] = {	/* 32/0, fn(x-x0), -2.821 < x <= -2.66 */
	FLIT(-111757730480459360.89),
	FLIT( 29138361921333636.227),
	FLIT(-7605635543471837.3634),
	FLIT( 1987570752795063.8740),
	FLIT(-520072613705173.85733),
	FLIT( 136270397145885.24852),
	FLIT(-35758721374667.469826),
	FLIT( 9398485347630.6486690),
	FLIT(-2474505336668.4851125),
	FLIT( 652740666207.52570348),
	FLIT(-172540549493.28646964),
	FLIT( 45711723541.795551153),
	FLIT(-12140906400.864337912),
	FLIT( 3233548126.0760565808),
	FLIT(-863873252.92932095220),
	FLIT( 231593359.80352937599),
	FLIT(-62330684.832407414773),
	FLIT( 16850484.270697366178),
	FLIT(-4578716.5945817174816),
	FLIT( 1251567.4465251002019),
	FLIT(-344497.59343483289168),
	FLIT( 95617.451643349703165),
	FLIT(-26804.578420618108710),
	FLIT( 7610.3979866583287295),
	FLIT(-2193.1127764806237429),
	FLIT( 646.90599971292827571),
	FLIT(-194.76615530344621664),
	FLIT( 62.627282713513713193),
	FLIT(-20.095134916842602581),
	FLIT( 9.5751894757096666708),
	FLIT(-1.9143501856115988164),
	FLIT( 0.63609282465313460000e-32),
	};

static const FTYPE sr10[] = {	/* 42/0, fn(x-x0), -2.66 < x <= -2.28 */
	FLIT( 2134602288151.7325973),
	FLIT( 1001825705477.8754020),
	FLIT( 468557430102.87333136),
	FLIT( 220358151446.77905511),
	FLIT( 103107284381.27778465),
	FLIT( 48612189337.342165762),
	FLIT( 22750853238.533041144),
	FLIT( 10759909800.736022102),
	FLIT( 5035014000.2995260339),
	FLIT( 2390700936.0723962024),
	FLIT( 1117953985.0661735268),
	FLIT( 533518885.54834525763),
	FLIT( 249118974.61233805307),
	FLIT( 119674503.86711416381),
	FLIT( 55731833.611605620914),
	FLIT( 27007615.310435627948),
	FLIT( 12522259.656340321558),
	FLIT( 6139385.8152913544239),
	FLIT( 2827023.6968710568926),
	FLIT( 1407979.5933782963805),
	FLIT( 641568.32774174789445),
	FLIT( 326433.70222156329746),
	FLIT( 146431.99829213393467),
	FLIT( 76721.036023984288730),
	FLIT( 33630.457055794339177),
	FLIT( 18347.170699977025256),
	FLIT( 7776.1581023442464729),
	FLIT( 4487.2184531889983114),
	FLIT( 1811.2054188018712620),
	FLIT( 1130.4578581775129886),
	FLIT( 425.17984568438211700),
	FLIT( 296.43641350761319567),
	FLIT( 100.64812354551159120),
	FLIT( 82.229156827042532426),
	FLIT( 24.038823062292954289),
	FLIT( 24.824942121894071095),
	FLIT( 5.8004145665998724718),
	FLIT( 8.7217825838153462180),
	FLIT( 1.4112911430779799506),
	FLIT( 4.8583209516339961204),
	FLIT( 1.5156034480216573216),
	FLIT( 0.20646555198525407000e-32),
	};

static const struct RootApprox rootapprox[] = {	/* table of roots */
	{FLIT(-7.00023), FLIT(-7.00017),
	FLIT(-7881522651463199.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(634929214277839.0) / FLIT(9007199254740992.0)
		/ FLIT(281474976710656.0),
	ROOTPOLY(sr00)},

	{FLIT(-6.99988), FLIT(-6.99976),
	FLIT(-7881075865650928.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(3423712940645379.0) / FLIT(9007199254740992.0)
		/ FLIT(9007199254740992.0) / FLIT(4.0),
	ROOTPOLY(sr01)},

	{FLIT(-6.00176), FLIT(-6.00137),
	FLIT(-6756959143951501.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(2602576852966713.0) / FLIT(9007199254740992.0)
		/ FLIT(4503599627370496.0),
	ROOTPOLY(sr02)},

	{FLIT(-5.9987), FLIT(-5.9985),
	FLIT(-6753831603008526.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(5648571687346535.0) / FLIT(9007199254740992.0)
		/ FLIT(1125899906842624.0),
	ROOTPOLY(sr03)},

	{FLIT(-5.00992), FLIT(-5.0066),
	FLIT(-5638752369161946.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(8561733107791203.0) / FLIT(9007199254740992.0)
		/ FLIT(1125899906842624.0),
	ROOTPOLY(sr04)},

	{FLIT(-4.9916), FLIT(-4.991),
	FLIT(-5619979645807357.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(6155472696447695.0) / FLIT(9007199254740992.0)
		/ FLIT(4503599627370496.0),
	ROOTPOLY(sr05)},

	{FLIT(-4.05), FLIT(-4.0338),
	FLIT(-4547917119067521.0) / FLIT(9007199254740992.0) * FLIT(8.0),
	FLIT(4288511774695039.0) / FLIT(9007199254740992.0)
		/ FLIT(2251799813685248.0),
	ROOTPOLY(sr06)},

	{FLIT(-3.957), FLIT(-3.943),
	FLIT(-8906530933714918.0) / FLIT(9007199254740992.0) * FLIT(4.0),
	FLIT(8601666998595293.0) / FLIT(9007199254740992.0)
		/ FLIT(2251799813685248.0),
	ROOTPOLY(sr07)},

	{FLIT(-3.177), FLIT(-3.121),
	FLIT(-7078714858690992.0) / FLIT(9007199254740992.0) * FLIT(4.0),
	FLIT(2290973324514835.0) / FLIT(9007199254740992.0)
		/ FLIT(1125899906842624.0),
	ROOTPOLY(sr08)},

	{FLIT(-2.821), FLIT(-2.66),
	FLIT(-6187231271966977.0) / FLIT(9007199254740992.0) * FLIT(4.0),
	FLIT(7346564866404229.0) / FLIT(9007199254740992.0)
		/ FLIT(9007199254740992.0),
	ROOTPOLY(sr09)},

	{FLIT(-2.66), FLIT(-2.28),
	FLIT(-5532727847745645.0) / FLIT(9007199254740992.0) * FLIT(4.0),
	FLIT(4127608264937587.0) / FLIT(9007199254740992.0)
		/ FLIT(1125899906842624.0),
	ROOTPOLY(sr10)},
	};

#endif												/* V10000K102 */
 #elif FBITS <= 71
  #define DECLARE(w, xinv)	FTYPE w = xinv * xinv;
  #define POLY(xinv, w)	(((((c[0] * w + c[1]) * w + c[2]) * w + c[3]) * w \
	+ c[4]) * xinv)

static const FTYPE c[] = {	/* Stirling fixup order 4, Gamma_big < x */
	FLIT( 0.0008417508417508417508417),
	FLIT(-0.0005952380952380952380951),
	FLIT( 0.0007936507936507936507938),
	FLIT(-0.0027777777777777777777777),
	FLIT( 0.0833333333333333333333333),
	};

/* Polynomial approximation fixup for 1.2 - 1.5
 */
static const FTYPE s10[] = {	/* 17/1, 1.2 <= x < 1.5 */
	FLIT( 0.1325721723559439893693539e-5),
	FLIT(-0.4032222300245615782012304e-5),
	FLIT( 0.9188960650664396325816485e-5),
	FLIT(-0.1899425284763590333521846e-4),
	FLIT( 0.3743393847743934420448662e-4),
	FLIT(-0.7207972071683309374576621e-4),
	FLIT( 0.1377376243623298720800729e-3),
	FLIT(-0.2644352696080985831563776e-3),
	FLIT( 0.5161143228230716773148459e-3),
	FLIT(-0.1037999984635054097547402e-2),
	FLIT( 0.2189603946946953839348604e-2),
	FLIT(-0.4972885954135403642372787e-2),
	FLIT( 0.1270516597579358135725326e-1),
	FLIT(-0.3986910503561781400545386e-1),
	FLIT( 0.1958258377112174870514865),
	FLIT( 0.4588681378719904821206167),
	FLIT(-0.1945755215190782738499777),
	FLIT(-0.1152408973524451317637435),
	};

static const FTYPE c10[] = {
	FLIT( 0.6998167377102503432518824),
	FLIT( 1.0000000000000000000151749),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.5), FLIT(12451552249753947341.0)
		/ FLIT(9223372036854775808.0), POLY1(s10), POLY1(c10)},
	};

/* Taylor expansions for negative roots
 */
static const FTYPE sr00[] = {	/* 3/0, lgamma(x-x0), -7.00024 < x <= -7.0001 */
	FLIT( 12710958.83395139409647909),
	FLIT( 5044.029941110829196534890),
	FLIT(-0.2398382181859361076763807e-34),
	};

static const FTYPE sr01[] = {	/* 3/0, lgamma(x-x0), -6.99988 < x <= -6.9998 */
	FLIT( 12690641.16604717924392577),
	FLIT(-5035.967373768125414506134),
	FLIT(-0.2777194204049808953659011e-35),
	};

static const FTYPE sr02[] = {	/* 3/0, lgamma(x-x0), -6.0015 < x <= -6.0011 */
	FLIT( 260548.4103030939696282634),
	FLIT( 723.7366299252801239787085),
	FLIT(-0.2821940614962839326908607e-35),
	};

static const FTYPE sr03[] = {	/* 3/0, lgamma(x-x0), -5.999 < x <= -5.9982 */
	FLIT( 257851.5896395626171594404),
	FLIT(-716.2454304275472920808369),
	FLIT( 0.5839389099023541205512310e-37),
	};

static const FTYPE sr04[] = {	/* 3/0, lgamma(x-x0), -5.009 < x <= -5.007 */
	FLIT( 7404.762432228682168278107),
	FLIT( 123.3621845633533933373067),
	FLIT(-0.4772120981532495930912661e-36),
	};

static const FTYPE sr05[] = {	/* 15/0, lgamma(x-x0), -4.992 < x <= -4.991 */
	FLIT( .7482089930210699257856737e28),
	FLIT(-.6813020277751575384000340e26),
	FLIT( 624070799300791608913548.3),
	FLIT(-5756446826254376299362.489),
	FLIT( 53540109714248596275.53527),
	FLIT(-503000968987169457.1189929),
	FLIT( 4784685740359657.139325350),
	FLIT(-46235700276520.07150420930),
	FLIT( 456096042262.1975859879914),
	FLIT(-4627747171.739249588193504),
	FLIT( 48911582.70905904446504353),
	FLIT(-551419.9978512883564523308),
	FLIT( 6995.235954894063841245896),
	FLIT(-116.5357816162436307966887),
	FLIT( 0.5417350506883166747364880e-37),
	};

static const FTYPE sr06[] = {	/* 23/0, lgamma(x-x0), -4.046 < x <= -4.032 */
	FLIT( .3680601217452123493305392e30),
	FLIT( .1517740559973266053622254e29),
	FLIT( .6272811372384941253290792e27),
	FLIT( .2599046273288770842163606e26),
	FLIT( 1079867563811784350321772.),
	FLIT( 45005901869396575262722.89),
	FLIT( 1882234790309111722308.495),
	FLIT( 79027439114890969640.02008),
	FLIT( 3332855778798614561.344502),
	FLIT( 141278668508269686.5944653),
	FLIT( 6024404000960524.098517006),
	FLIT( 258689045256062.4222615249),
	FLIT( 11200724416202.00293485747),
	FLIT( 489867910498.2001723423136),
	FLIT( 21692364960.71330968349785),
	FLIT( 975830163.5382145099019070),
	FLIT( 44812215.95015570915100009),
	FLIT( 2116669.417577535056382773),
	FLIT( 104145.5552014554115150904),
	FLIT( 5465.691820777134250040684),
	FLIT( 324.2553229378471572043328),
	FLIT( 26.79048088614059326062263),
	FLIT(-0.1973209827313205801374555e-37),
	};

static const FTYPE sr07[] = {	/* 27/0, lgamma(x-x0), -3.965 < x <= -3.946 */
	FLIT( .4737916934294885715397183e34),
	FLIT(-.2202844434213406419269112e33),
	FLIT( .1025830581007995444549887e32),
	FLIT(-.4785442407431209689837939e30),
	FLIT( .2236610171418506941513084e29),
	FLIT(-.1047506504060743344681883e28),
	FLIT( .4917100374827080041530452e26),
	FLIT(-2313920933460008349805858.),
	FLIT( 109192461783823105408088.0),
	FLIT(-5168675744572264565955.666),
	FLIT( 245511179588567033216.1216),
	FLIT(-11707469715963823367.93214),
	FLIT( 560775863730503686.5438574),
	FLIT(-26998338793041111.22525566),
	FLIT( 1307561713653824.542589688),
	FLIT(-63769616182266.90388003412),
	FLIT( 3135952925293.292291579878),
	FLIT(-155772242416.5721681402191),
	FLIT( 7834398183.738318402234743),
	FLIT(-400276998.1874636580544444),
	FLIT( 20877114.72650332924808826),
	FLIT(-1119991.501665522135393842),
	FLIT( 62588.08191876606055936682),
	FLIT(-3730.604715680612510741850),
	FLIT( 251.7146825868894001461405),
	FLIT(-20.72506084580370567830050),
	FLIT(-0.3914441799099178801913323e-37),
	};

static const FTYPE sr08[] = {	/* 25/0, lgamma(x-x0), -3.175 < x <= -3.115 */
	FLIT( 7070815309148457255.897609),
	FLIT( 1059374549682964561.906512),
	FLIT( 159019845254241963.3815963),
	FLIT( 23919458772922345.04650279),
	FLIT( 3606095996440261.253052972),
	FLIT( 545017333309682.2439767331),
	FLIT( 82601521376228.04505912892),
	FLIT( 12557646866390.60185075713),
	FLIT( 1915727973453.940358529405),
	FLIT( 293399385882.3304505170777),
	FLIT( 45135583357.51527864039421),
	FLIT( 6979115396.671345966273222),
	FLIT( 1085573221.570005082357655),
	FLIT( 170037345.8217431236491050),
	FLIT( 26855525.05557588804934825),
	FLIT( 4284377.443645979621552236),
	FLIT( 692049.6022897746119868314),
	FLIT( 113559.6642371351622398374),
	FLIT( 19023.06198435635837184406),
	FLIT( 3277.193785495341744981610),
	FLIT( 588.8907422380013467336813),
	FLIT( 112.2689862971760025782803),
	FLIT( 25.83133837238795893574727),
	FLIT( 7.781884658131350872139553),
	FLIT( 0.2972230940579952406886192e-37),
	};

static const FTYPE sr09[] = {	/* 40/0, lgamma(x-x0), -2.84 < x <= -2.70 */
	FLIT(-5407514558725657118737.582),
	FLIT( 1400315281273289638308.010),
	FLIT(-362873138640144268947.7751),
	FLIT( 94102500746381141738.20757),
	FLIT(-24422085179648813742.74607),
	FLIT( 6343354596689220736.086400),
	FLIT(-1649039607636613316.405086),
	FLIT( 429083850154303492.7006230),
	FLIT(-111757730480459360.8871141),
	FLIT( 29138361921333636.22728055),
	FLIT(-7605635543471837.363364682),
	FLIT( 1987570752795063.873972719),
	FLIT(-520072613705173.8573270289),
	FLIT( 136270397145885.2485228167),
	FLIT(-35758721374667.46982641961),
	FLIT( 9398485347630.648668984659),
	FLIT(-2474505336668.485112416302),
	FLIT( 652740666207.5257034736219),
	FLIT(-172540549493.2864696524439),
	FLIT( 45711723541.79555115271518),
	FLIT(-12140906400.86433791050664),
	FLIT( 3233548126.076056580714905),
	FLIT(-863873252.9293209521982578),
	FLIT( 231593359.8035293759902635),
	FLIT(-62330684.83240741477259405),
	FLIT( 16850484.27069736617663109),
	FLIT(-4578716.594581717481629054),
	FLIT( 1251567.446525100201877994),
	FLIT(-344497.5934348328916728255),
	FLIT( 95617.45164334970316441555),
	FLIT(-26804.57842061810870992989),
	FLIT( 7610.397986658328729438846),
	FLIT(-2193.112776480623742905942),
	FLIT( 646.9059997129282757212712),
	FLIT(-194.7661553034462166478795),
	FLIT( 62.62728271351371319293175),
	FLIT(-20.09513491684260258245915),
	FLIT( 9.575189475709666670650146),
	FLIT(-1.914350185611598816494731),
	FLIT( 0.5610274133193390970497697e-38),
	};

static const FTYPE sr10[] = {	/* 30/0, lgamma(x-x0), -2.70 < x <= -2.60 */
	FLIT(-574948187218.5304529258019),
	FLIT( 208418727380.5306276773109),
	FLIT(-75648272413.99250821722152),
	FLIT( 27495241802.33235506892543),
	FLIT(-10008265089.79028859473625),
	FLIT( 3648848628.640397393689314),
	FLIT(-1332621633.564021787864170),
	FLIT( 487619283.6721054968716464),
	FLIT(-178793115.4947542175205434),
	FLIT( 65706894.35823419034316735),
	FLIT(-24207512.79633650470970575),
	FLIT( 8943530.355326904733889325),
	FLIT(-3314230.008819462831299282),
	FLIT( 1232573.985584086852108328),
	FLIT(-460095.2960052307652890919),
	FLIT( 172581.4637575116438716516),
	FLIT(-65017.92853383537950059881),
	FLIT( 24675.17271525389431048177),
	FLIT(-9405.449035366033712924981),
	FLIT( 3632.529945448044614970675),
	FLIT(-1404.401370991285060023976),
	FLIT( 559.0295454105787415848951),
	FLIT(-219.1364581915463941193382),
	FLIT( 92.91246727344536615957131),
	FLIT(-36.38586475697763733631357),
	FLIT( 18.18589775328140903990048),
	FLIT(-6.649739481509334418828351),
	FLIT( 6.058515918076041501084369),
	FLIT(-0.4491902347377631807527278),
	FLIT(-0.1099707846719332687791280),
	};

static const FTYPE sr11[] = {	/* 40/0, lgamma(x-x0), -2.60 < x <= -2.34 */
	FLIT( 468557430102.8733313621165),
	FLIT( 220358151446.7790551231844),
	FLIT( 103107284381.2777846420778),
	FLIT( 48612189337.34216576056940),
	FLIT( 22750853238.53304114531696),
	FLIT( 10759909800.73602210092743),
	FLIT( 5035014000.299526033924835),
	FLIT( 2390700936.072396202535509),
	FLIT( 1117953985.066173526665866),
	FLIT( 533518885.5483452576316036),
	FLIT( 249118974.6123380530754082),
	FLIT( 119674503.8671141638212242),
	FLIT( 55731833.61160562091541061),
	FLIT( 27007615.31043562794698138),
	FLIT( 12522259.65634032155674934),
	FLIT( 6139385.815291354423921095),
	FLIT( 2827023.696871056892602729),
	FLIT( 1407979.593378296380370927),
	FLIT( 641568.3277417478944361353),
	FLIT( 326433.7022215632974638549),
	FLIT( 146431.9982921339346753070),
	FLIT( 76721.03602398428872955490),
	FLIT( 33630.45705579433917837952),
	FLIT( 18347.17069997702525607778),
	FLIT( 7776.158102344246472940337),
	FLIT( 4487.218453188998311453348),
	FLIT( 1811.205418801871261981939),
	FLIT( 1130.457858177512988623272),
	FLIT( 425.1798456843821169994825),
	FLIT( 296.4364135076131956819299),
	FLIT( 100.6481235455115911968425),
	FLIT( 82.22915682704253242636426),
	FLIT( 24.03882306229295428853727),
	FLIT( 24.82494212189407109434820),
	FLIT( 5.800414566599872471733794),
	FLIT( 8.721782583815346217950569),
	FLIT( 1.411291143077979950553442),
	FLIT( 4.858320951633996120493900),
	FLIT( 1.515603448021657321637057),
	FLIT(-0.2000595528853219903023305e-39),
	};

static const struct RootApprox rootapprox[] = {	/* table of roots */
	{FLIT(-7.00024), FLIT(-7.0001),
	FLIT(-16141358390196630975.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(8573242119178660507.0) / FLIT(18446744073709551616.0) 
		/ FLIT(2305843009213693952.0),
	ROOTPOLY(sr00)},

	{FLIT(-6.99988), FLIT(-6.9998),
	FLIT(-16140443372853100520.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(753880598405232515.0) / FLIT(18446744073709551616.0) 
		/ FLIT(288230376151711744.0),
	ROOTPOLY(sr01)},

	{FLIT(-6.0015), FLIT(-6.0011),
	FLIT(-13838252326812673901.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(17328247341119915663.0) / FLIT(18446744073709551616.0) 
		/ FLIT(2305843009213693952.0),
	ROOTPOLY(sr02)},

	{FLIT(-5.999), FLIT(-5.9982),
	FLIT(-13831847122961459964.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(12414863762514018759.0) / FLIT(18446744073709551616.0) 
		/ FLIT(4611686018427387904.0),
	ROOTPOLY(sr03)},

	{FLIT(-5.009), FLIT(-5.007),
	FLIT(-11548164852043663462.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(13147453502287921767.0) / FLIT(18446744073709551616.0) 
		/ FLIT(2305843009213693952.0),
	ROOTPOLY(sr04)},

	{FLIT(-4.992), FLIT(-4.991),
	FLIT(-11509718314613466787.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(16567256425704677317.0) / FLIT(18446744073709551616.0) 
		/ FLIT(2305843009213693952.0),
	ROOTPOLY(sr05)},

	{FLIT(-4.046), FLIT(-4.032),
	FLIT(-9314134259850282521.0)
		/ FLIT(18446744073709551616.0) * FLIT(8.0),
	FLIT(10096681428698638973.0) / FLIT(18446744073709551616.0) 
		/ FLIT(2305843009213693952.0),
	ROOTPOLY(sr06)},

	{FLIT(-3.965), FLIT(-3.946),
	FLIT(-18240575352248150109.0)
		/ FLIT(18446744073709551616.0) * FLIT(4.0),
	FLIT(1827704346757077479.0) / FLIT(18446744073709551616.0) 
		/ FLIT(576460752303423488.0),
	ROOTPOLY(sr07)},

	{FLIT(-3.175), FLIT(-3.115),
	FLIT(-14497208030599150575.0)
		/ FLIT(18446744073709551616.0) * FLIT(4.0),
	FLIT(15016577080098907823.0) / FLIT(18446744073709551616.0) 
		/ FLIT(4611686018427387904.0),
	ROOTPOLY(sr08)},

	{FLIT(-2.84), FLIT(-2.70),
	FLIT(-12671449644988368479.0)
		/ FLIT(18446744073709551616.0) * FLIT(4.0),
	FLIT(11139322617798086701.0) / FLIT(18446744073709551616.0) 
		/ FLIT(4611686018427387904.0),
	ROOTPOLY(sr09)},

	{FLIT(-2.70), FLIT(-2.60),
	FLIT(-12220967948832577946.0)
		/ FLIT(18446744073709551616.0) * FLIT(4.0),
	FLIT(14757395258967641293.0) / FLIT(18446744073709551616.0) 
		/ FLIT(9223372036854775808.0),
	ROOTPOLY(sr10)},

	{FLIT(-2.60), FLIT(-2.34),
	FLIT(-11331026632183079083.0)
		/ FLIT(18446744073709551616.0) * FLIT(4.0),
	FLIT(5585372299732956403.0) / FLIT(18446744073709551616.0) 
		/ FLIT(18446744073709551616.0) / FLIT(4.0),
	ROOTPOLY(sr11)},
	};

 #elif FBITS <= 117
  #define DECLARE(w, xinv)	FTYPE w = xinv * xinv;
  #define POLY(xinv, w)	(FNAME(Poly)(w, c, \
	sizeof (c) / sizeof (c[0]) - 1) * xinv)

static const FTYPE c[] = {	/* Stirling fixup order 10, Gamma_big < x */
	FLIT(13.40286404416839199447895100069013111),
	FLIT(-1.39243221690590111642743221690590111),
	FLIT( 0.17964437236883057316493849001588940),
	FLIT(-0.02955065359477124183006535947712417),
	FLIT( 0.00641025641025641025641025641025642),
	FLIT(-0.00191752691752691752691752691752691),
	FLIT( 0.00084175084175084175084175084175085),
	FLIT(-0.00059523809523809523809523809523810),
	FLIT( 0.00079365079365079365079365079365079),
	FLIT(-0.00277777777777777777777777777777777),
	FLIT( 0.08333333333333333333333333333333333),
	};

/* Polynomial approximation fixup for 1.2 - 1.5
 */
static const FTYPE s10[] = {	/* 30/1, 1.2 <= x < 1.5 */
	FLIT(-0.50319865904335514758104988981876759e-8),
	FLIT( 0.14431338535755456689279900386236858e-7),
	FLIT(-0.30384531158720828751061994738879497e-7),
	FLIT( 0.57949035457902346176604845878785942e-7),
	FLIT(-0.10464925926606684687211297820001277e-6),
	FLIT( 0.18253374500190415893279193626737762e-6),
	FLIT(-0.31111594209958892168549040948815332e-6),
	FLIT( 0.52192585637109981233563761282563538e-6),
	FLIT(-0.86598268886862915766821806920488350e-6),
	FLIT( 0.14260815025002117791297828355426433e-5),
	FLIT(-0.23371363713745371401130355999766978e-5),
	FLIT( 0.38201900601151965313378355380254368e-5),
	FLIT(-0.62399278289290786409613402176250449e-5),
	FLIT( 0.10203288160986861285638131861232684e-4),
	FLIT(-0.16731367457884506616372797071041674e-4),
	FLIT( 0.27565178768941059968509351578327591e-4),
	FLIT(-0.45722931594923527855661369208955208e-4),
	FLIT( 0.76546408561431983507259004542030011e-4),
	FLIT(-0.12973834887458884413580159155748793e-3),
	FLIT( 0.22350931652187239582225774029507524e-3),
	FLIT(-0.39348868136025186502166730356247785e-3),
	FLIT( 0.71319219682769245935762359227035709e-3),
	FLIT(-0.13450903405408233231605651415939042e-2),
	FLIT( 0.26821612668317646432770713217031178e-2),
	FLIT(-0.57987787041563128599690454350147392e-2),
	FLIT( 0.14197159000045117245565364385010590e-1),
	FLIT(-0.42979634374007330238001843267532936e-1),
	FLIT( 0.20507698990452611614634120972381803),
	FLIT( 0.45691126540029448906974693964430300),
	FLIT(-0.19655494461557207361524452814121165),
	FLIT(-0.11524089735244513175952417708821399),
	};

static const FTYPE c10[] = {
	FLIT( 0.71699313204567505953697217524042305),
	FLIT( 1.0),
//	FLIT( 1.00000000000000000000000000000000006),
	};

static const struct Approx approx[] = {	/* table of approximations */
	{FLIT(1.5), FLIT(3504800379511008649258085022223565.0)
		/ FLIT(2596148429267413814265248164610048.0), POLY1(s10), POLY1(c10)},
	};

/* Taylor expansions for negative roots
 */
static const FTYPE sr00[] = {	/* -7.00024 < x <= -7.0001 */
	FLIT( 651704366027372303.65656567734322592),
	FLIT( 161568434353282.84644813273229360130),
	FLIT( 42725890801.879194202328443026548147),
	FLIT( 12710958.833951394096479091371862116),
	FLIT( 5044.0299411108291965348895970389542),
	FLIT(-0.48717573785944491635241398300678455e-64),
	};

static const FTYPE sr01[] = {	/* -6.99988 < x <= -6.9998 */
	FLIT(-649103215888582942.59437137584913985),
	FLIT( 161052333144473.79179080279743346254),
	FLIT(-42623489764.758116459934164737639928),
	FLIT( 12690641.166047179243925778374512240),
	FLIT(-5035.9673737681254145061352361561094),
	FLIT(-0.11879194274927111327769972003826428e-64),
	};

static const FTYPE sr02[] = {	/* -6.0015 < x <= -6.0011 */
	FLIT( 39203019565186.829429659384353471698),
	FLIT( 67884656938.770961214132155463873959),
	FLIT( 125386984.94090366665953046938350334),
	FLIT( 260548.41030309396962826348013658214),
	FLIT( 723.73662992528012397870838742930395),
	FLIT( 0.25698218675919302666296906735558655e-64),
	};

static const FTYPE sr03[] = {	/* -5.999 < x <= -5.9982 */
	FLIT(-38196442388133.234370073016104879495),
	FLIT( 66486633581.993339903469888023778990),
	FLIT(-123445282.15783688644320476215899832),
	FLIT( 257851.58963956261715944054314115408),
	FLIT(-716.24543042754729208083690850643731),
	FLIT(-0.24232006371804012796229602642737737e-64),
	};

static const FTYPE sr04[] = {	/* -5.009 < x <= -5.007 */
	FLIT( 5335250743.3843846461673574078072725),
	FLIT( 54807486.356655854741891212612688197),
	FLIT( 600556.16845281092209665075149129638),
	FLIT( 7404.7624322286821682781082575223511),
	FLIT( 123.36218456335339333730679831563451),
	FLIT(-0.41772127001754462153531726360138677e-65),
	};

static const FTYPE sr05[] = {	/* -4.992 < x <= -4.991 */
	FLIT(-.26531275744826903882877857697671662e51),
	FLIT( .23367861752395735230289353861744158e49),
	FLIT(-.20617426482742857738639549336320788e47),
	FLIT( .18225174034921744857018005365958488e45),
	FLIT(-.16143851147908913208511629432508193e43),
	FLIT( .14332716741068884527539200298044289e41),
	FLIT(-.12756660189132509223512177445364856e39),
	FLIT( .11385448866091413450599885410178145e37),
	FLIT(-10193088975670586996549325171499893.),
	FLIT( 91572870536004151854016135791098.477),
	FLIT(-825900304352139192654793726439.90620),
	FLIT( 7482089930210699257856738039.1129217),
	FLIT(-68130202777515753840003404.138888077),
	FLIT( 624070799300791608913548.27003045870),
	FLIT(-5756446826254376299362.4891348688662),
	FLIT( 53540109714248596275.535266757261757),
	FLIT(-503000968987169457.11899287373403289),
	FLIT( 4784685740359657.1393253498606932259),
	FLIT(-46235700276520.071504209295497623980),
	FLIT( 456096042262.19758598799154902738162),
	FLIT(-4627747171.7392495881935052472276561),
	FLIT( 48911582.709059044465043527586289669),
	FLIT(-551419.99785128835645233066591266173),
	FLIT( 6995.2359548940638412458958571965662),
	FLIT(-116.53578161624363079668877957442926),
	FLIT(-0.20475790152868741484779097637520836e-65),
	};

static const FTYPE sr06[] = {	/* -4.046 < x <= -4.032 */
	FLIT( .39348162513521850126536990572902620e55),
	FLIT( .15885292994275360638979589579536913e54),
	FLIT( .64172894541614356866743607405509733e52),
	FLIT( .25942324660559221157125173264421604e51),
	FLIT( .10495025598355183312074129895695384e50),
	FLIT( .42490647326601138030497308788373929e48),
	FLIT( .17217015226160910256952147696677955e47),
	FLIT( .69822955760139349130816001135582941e45),
	FLIT( .28342462446799886204373706015978760e44),
	FLIT( .11515989312858030807318198609287875e43),
	FLIT( .46840020997933494022262644451497312e41),
	FLIT( .19072855861696968241069560720145084e40),
	FLIT( .77755493495899237157082606604456219e38),
	FLIT( .31739548025960890797838219940140003e37),
	FLIT( .12973780414055209382076560142595356e36),
	FLIT( 5310987401064355749843615719162447.3),
	FLIT( 217760661400735009111965583704610.73),
	FLIT( 8944132440885960454208505271950.7917),
	FLIT( 368060121745212349330539147943.59464),
	FLIT( 15177405599732660536222553538.944803),
	FLIT( 627281137238494125329079087.29413148),
	FLIT( 25990462732887708421636055.565061437),
	FLIT( 1079867563811784350321772.2396372325),
	FLIT( 45005901869396575262722.886574968380),
	FLIT( 1882234790309111722308.4951554866367),
	FLIT( 79027439114890969640.020078194616554),
	FLIT( 3332855778798614561.3445015108070564),
	FLIT( 141278668508269686.59446527166346212),
	FLIT( 6024404000960524.0985170057087807360),
	FLIT( 258689045256062.42226152492476779414),
	FLIT( 11200724416202.002934857482518611408),
	FLIT( 489867910498.20017234231364872437694),
	FLIT( 21692364960.713309683497840247471775),
	FLIT( 975830163.53821450990190703040514311),
	FLIT( 44812215.950155709151000090707365608),
	FLIT( 2116669.4175775350563827726991045794),
	FLIT( 104145.55520145541151509050304726367),
	FLIT( 5465.6918207771342500406851778988785),
	FLIT( 324.25532293784715720433271573720736),
	FLIT( 26.790480886140593260622633936281014),
	FLIT(-0.17033942698284077488662163307915700e-66),
	};

static const FTYPE sr07[] = {	/* -3.9649 < x <= -3.946 */
	FLIT(-.57655262706125382505897195996670280e62),
	FLIT( .26335527890067163709501345918581525e61),
	FLIT(-.12035119103507208629040551381243511e60),
	FLIT( .55026675647819953057416540545808005e58),
	FLIT(-.25172163491709224435155805926464419e57),
	FLIT( .11521333698520381189247448055113810e56),
	FLIT(-.52763213218911508378751589126398813e54),
	FLIT( .24177876096124611039758698305098100e53),
	FLIT(-.11086043502332935685114774438070142e52),
	FLIT( .50865185819892145696950963666468988e50),
	FLIT(-.23354230080915168918109583507235545e49),
	FLIT( .10730694339370502415649533469868101e48),
	FLIT(-.49342974624550632164833543789844440e46),
	FLIT( .22707927609650205473929000626186989e45),
	FLIT(-.10459369959918922879874228502323084e44),
	FLIT( .48220591442375691069741165722519938e42),
	FLIT(-.22252758967460935073212747160574835e41),
	FLIT( .10279863536265753525717638342099751e40),
	FLIT(-.47541584580494786876805438320516799e38),
	FLIT( .22012869853988470788159402433961887e37),
	FLIT(-.10205492776237272729529373615741507e36),
	FLIT( 4737916934294885715397183233232625.3),
	FLIT(-220284443421340641926911134294573.87),
	FLIT( 10258305810079954445498877906105.941),
	FLIT(-478544240743120968983793857133.45153),
	FLIT( 22366101714185069415130853046.928018),
	FLIT(-1047506504060743344681882741.5083328),
	FLIT( 49171003748270800415304514.844583419),
	FLIT(-2313920933460008349805858.4832700208),
	FLIT( 109192461783823105408088.00179811028),
	FLIT(-5168675744572264565955.6661773887516),
	FLIT( 245511179588567033216.12158399887180),
	FLIT(-11707469715963823367.932139986369394),
	FLIT( 560775863730503686.54385754234418754),
	FLIT(-26998338793041111.225255658918679331),
	FLIT( 1307561713653824.5425896874147597126),
	FLIT(-63769616182266.903880034105304686172),
	FLIT( 3135952925293.2922915798767400057579),
	FLIT(-155772242416.57216814021924936704102),
	FLIT( 7834398183.7383184022347429432928155),
	FLIT(-400276998.18746365805444454324967249),
	FLIT( 20877114.726503329248088260089496786),
	FLIT(-1119991.5016655221353938405104977730),
	FLIT( 62588.081918766060559366806027424570),
	FLIT(-3730.6047156806125107418498987900975),
	FLIT( 251.71468258688940014614044664622881),
	FLIT(-20.725060845803705678300499820071072),
	FLIT( 0.33892782621628092172184492425518400e-66),
	};

static const FTYPE sr08[] = {	/* -3.175 < x <= -3.115 */
	FLIT( 40867323448314495482545984061947390.),
	FLIT( 6007475333890083852915973734007135.2),
	FLIT( 883596660916641823564581579023048.79),
	FLIT( 130039283355424726711464880338830.07),
	FLIT( 19149903409811956820198303741960.436),
	FLIT( 2821916989275984387161624566119.4034),
	FLIT( 416123979201772583215360148144.03341),
	FLIT( 61407102003005775929984079396.055697),
	FLIT( 9068797292491963687287680346.2553455),
	FLIT( 1340403205978416266850984180.7685569),
	FLIT( 198288291639164803346619108.30061457),
	FLIT( 29360109346168431176292352.303464906),
	FLIT( 4351536084619783781268392.6884419056),
	FLIT( 645623997273925970387981.98024304635),
	FLIT( 95895793519298684001703.259406332150),
	FLIT( 14260546194760278688861.918131229265),
	FLIT( 2123376775851445909435.7689827507410),
	FLIT( 316602336231485053344.60551318369253),
	FLIT( 47276366477387529802.163620247529484),
	FLIT( 7070815309148457255.8976091226588401),
	FLIT( 1059374549682964561.9065123202979946),
	FLIT( 159019845254241963.38159627280876351),
	FLIT( 23919458772922345.046502791487504738),
	FLIT( 3606095996440261.2530529712152640867),
	FLIT( 545017333309682.24397673320176709709),
	FLIT( 82601521376228.045059128905968630859),
	FLIT( 12557646866390.601850757132089424785),
	FLIT( 1915727973453.9403585294044268096593),
	FLIT( 293399385882.33045051707776023506199),
	FLIT( 45135583357.515278640394217294211233),
	FLIT( 6979115396.6713459662732211746875243),
	FLIT( 1085573221.5700050823576544353278495),
	FLIT( 170037345.82174312364910499378477356),
	FLIT( 26855525.055575888049348243177952387),
	FLIT( 4284377.4436459796215522362516500652),
	FLIT( 692049.60228977461198683151129728654),
	FLIT( 113559.66423713516223983750433004736),
	FLIT( 19023.061984356358371844057348787475),
	FLIT( 3277.1937854953417449816102759392465),
	FLIT( 588.89074223800134673368126056670801),
	FLIT( 112.26898629717600257828030157934106),
	FLIT( 25.831338372387958935747275217113303),
	FLIT( 7.7818846581313508721395525959964645),
	FLIT(-0.96285321346215546152644236614956000e-68),
	};

static const FTYPE sr09[] = {	/* -2.82 < x <= -2.78 */
	FLIT(-46640753937073242970001848.883839493),
	FLIT( 9573628439715034083316327.3638379134),
	FLIT(-1966475030860385379275656.0308455925),
	FLIT( 404219867454634772406769.62470338478),
	FLIT(-83153801304953438895018.851753262313),
	FLIT( 17119900268666884478458.739832195646),
	FLIT(-3527737025058630740955.9782008324550),
	FLIT( 727595761418342590371.47438668314289),
	FLIT(-150213318486367502487.58908231951993),
	FLIT( 31044085820515950547.759925989619078),
	FLIT(-6422914307692955257.8886276437171939),
	FLIT( 1330460820879255040.7854981869363757),
	FLIT(-275947429515697322.64507308061070096),
	FLIT( 57312158437875613.687730028837255107),
	FLIT(-11920928955078114.412507446666107130),
	FLIT( 2483526865641284.8654506950874104931),
	FLIT(-518301258916432.85254365894031262245),
	FLIT( 108372081409807.29742672441328113156),
	FLIT(-22706531343000.790548341366110234293),
	FLIT( 4768371582035.5881132919672004419946),
	FLIT(-1003867701476.6127551586432652821734),
	FLIT( 211927625871.14159657661966946590808),
	FLIT(-44878791357.684476941095508500368296),
	FLIT( 9536743166.2863319016028683771711666),
	FLIT(-2034505206.4428703303755367699589263),
	FLIT( 435965403.41539554414429036541471872),
	FLIT(-93900238.992547977344875452851100158),
	FLIT( 20345053.305418276156127558482281892),
	FLIT(-4438919.4083328946651886870750845913),
	FLIT( 976563.44779519035206156282506957916),
	FLIT(-217013.08210554139878292266036144983),
	FLIT( 48828.900537221600654758005830914286),
	FLIT(-11160.071141820360609183264863911006),
	FLIT( 2604.8651812679373285670830390486428),
	FLIT(-624.46300606719138780718797805948621),
	FLIT( 157.02344196479597018413564713460284),
	FLIT(-41.189623785765850522537299533967159),
	FLIT( 14.133035100570292181958615319660072),
	FLIT(-3.1263414838937755971345095497515395),
	FLIT( 0.12980129166171703253579491194210428),
	};

static const FTYPE sr10[] = {	/* -2.78 < x <= -2.69 */
	FLIT(-4115238598637870704004960271.1058670),
	FLIT( 1059978321945447024301985695.4703102),
	FLIT(-273141369926393603677383233.52199756),
	FLIT( 70416531605795385144380259.450090734),
	FLIT(-18162142056552229991104661.042614557),
	FLIT( 4686774150139778898375605.3343523638),
	FLIT(-1210055715204292168926666.4880021448),
	FLIT( 312587532881588955646589.87255988243),
	FLIT(-80794948205682859171527.170010318847),
	FLIT( 20895616676280025422884.194254896332),
	FLIT(-5407514558725657118737.5821497184698),
	FLIT( 1400315281273289638308.0102709307603),
	FLIT(-362873138640144268947.77514094459188),
	FLIT( 94102500746381141738.207568489128468),
	FLIT(-24422085179648813742.746072107653024),
	FLIT( 6343354596689220736.0864001349197646),
	FLIT(-1649039607636613316.4050855490292131),
	FLIT( 429083850154303492.70062298232283255),
	FLIT(-111757730480459360.88711424881585416),
	FLIT( 29138361921333636.227280539802397692),
	FLIT(-7605635543471837.3633646806123363236),
	FLIT( 1987570752795063.8739727187568480200),
	FLIT(-520072613705173.85732702889782161066),
	FLIT( 136270397145885.24852281678616058396),
	FLIT(-35758721374667.469826419618464637956),
	FLIT( 9398485347630.6486689846588780135090),
	FLIT(-2474505336668.4851124163012114757295),
	FLIT( 652740666207.52570347362192645703315),
	FLIT(-172540549493.28646965244393648039473),
	FLIT( 45711723541.795551152715168226843677),
	FLIT(-12140906400.864337910506647945563493),
	FLIT( 3233548126.0760565807149043366986839),
	FLIT(-863873252.92932095219825766094758461),
	FLIT( 231593359.80352937599026344863036466),
	FLIT(-62330684.832407414772594035202047300),
	FLIT( 16850484.270697366176631085192086168),
	FLIT(-4578716.5945817174816290548201769448),
	FLIT( 1251567.4465251002018779946225735025),
	FLIT(-344497.59343483289167282538300575725),
	FLIT( 95617.451643349703164415542060544590),
	FLIT(-26804.578420618108709929894055406546),
	FLIT( 7610.3979866583287294388455623511819),
	FLIT(-2193.1127764806237429059405445602350),
	FLIT( 646.90599971292827572127108543985399),
	FLIT(-194.76615530344621664787938610388652),
	FLIT( 62.627282713513713192931735935896934),
	FLIT(-20.095134916842602582459140657039975),
	FLIT( 9.5751894757096666706501462395944683),
	FLIT(-1.9143501856115988164947315746096804),
	FLIT( 0.21026009645048796278277695276305000e-67),
	};

static const FTYPE sr11[] = {	/* -2.69 < x <= -2.62 */
	FLIT(-15498214354841153.067515076922900200),
	FLIT( 5567121735878123.6857745438360756842),
	FLIT(-2001154569576149.2046949792915266908),
	FLIT( 719859769010308.97476191595445438951),
	FLIT(-259149516688454.25723203959521861000),
	FLIT( 93370046557813.553011892682736584920),
	FLIT(-33669804598246.023713443415090188459),
	FLIT( 12152695143814.131183064296208591485),
	FLIT(-4390651117442.4633474557979029702130),
	FLIT( 1587952175158.3988672197859876530133),
	FLIT(-574948187218.53045292580187949694311),
	FLIT( 208418727380.53062767731091136376533),
	FLIT(-75648272413.992508217221509855021242),
	FLIT( 27495241802.332355068925433394646143),
	FLIT(-10008265089.790288594736241639886218),
	FLIT( 3648848628.6403973936893152899036096),
	FLIT(-1332621633.5640217878641700822183448),
	FLIT( 487619283.67210549687164649023179204),
	FLIT(-178793115.49475421752054345764322217),
	FLIT( 65706894.358234190343167344060843999),
	FLIT(-24207512.796336504709705739161161114),
	FLIT( 8943530.3553269047338893239412451974),
	FLIT(-3314230.0088194628312992814238321006),
	FLIT( 1232573.9855840868521083267991091765),
	FLIT(-460095.29600523076528909186702064302),
	FLIT( 172581.46375751164387165156217988953),
	FLIT(-65017.928533835379500598821872634314),
	FLIT( 24675.172715253894310481777091829297),
	FLIT(-9405.4490353660337129249815489852185),
	FLIT( 3632.5299454480446149706735423553178),
	FLIT(-1404.4013709912850600239755653610753),
	FLIT( 559.02954541057874158489520966925356),
	FLIT(-219.13645819154639411933810829092423),
	FLIT( 92.912467273445366159571321850396188),
	FLIT(-36.385864756977637336313580250705884),
	FLIT( 18.185897753281409039900465004087537),
	FLIT(-6.6497394815093344188283521761765971),
	FLIT( 6.0585159180760415010843692641050185),
	FLIT(-0.44919023473776318075272769591275843),
	FLIT(-0.10997078467193326877912801144326197),
	};

static const FTYPE sr12[] = {	/* -2.62 < x <= -2.56 */
	FLIT(-5846000364.3341737476782485751946830),
	FLIT( 2482620107.2460867885841416713237505),
	FLIT(-1055476733.9007583731160457641253402),
	FLIT( 449448676.29390948621436567064826517),
	FLIT(-191608615.48282826338639301510620394),
	FLIT( 81855155.625435940840004075974149628),
	FLIT(-35006036.079603226662709095659495277),
	FLIT( 15013331.915001246288203171868263727),
	FLIT(-6443347.0739703085230596300264623717),
	FLIT( 2777104.8923381448146560378548272697),
	FLIT(-1196525.3257893116978184643685131434),
	FLIT( 519084.26984407984481201674674308560),
	FLIT(-224559.82459945625404387832182136032),
	FLIT( 98315.229317354951136743503233041027),
	FLIT(-42687.300755041051029745340947869816),
	FLIT( 18947.397207896481859845450806323573),
	FLIT(-8241.8030370599786570577332262916403),
	FLIT( 3740.1213030374883496210942372845069),
	FLIT(-1621.7594718279550609539882509271311),
	FLIT( 764.57806415283509824286336255760367),
	FLIT(-326.57033068535895636525897005878259),
	FLIT( 165.06944705353738737058341112203713),
	FLIT(-67.619649042965202270859983288035081),
	FLIT( 39.071286207991799001091055918199559),
	FLIT(-14.482866841321904422730988162926364),
	FLIT( 11.028862402974199750302423862864689),
	FLIT(-3.2731893550740109025534369666166893),
	FLIT( 5.1908803469954533018910271142550573),
	FLIT( 0.21974193396593693445334183074503459),
	FLIT(-0.11633668646646775324939982052967264),
	};

static const FTYPE sr13[] = {	/* -2.56 < x <= -2.3514 */
	FLIT( 938998144547532.84360411509653219549),
	FLIT( 438292227747877.60429065223540324039),
	FLIT( 204457904890073.86857398933253672312),
	FLIT( 95537146952093.171464302094680775642),
	FLIT( 44597895480216.151476797507718796433),
	FLIT( 20865139988016.201661081303624310956),
	FLIT( 9746780409948.6884982523382191837033),
	FLIT( 4566622151042.8512825979090367787019),
	FLIT( 2134602288151.7325972644761731657211),
	FLIT( 1001825705477.8754020393075410049177),
	FLIT( 468557430102.87333136211636744422324),
	FLIT( 220358151446.77905512318445606673064),
	FLIT( 103107284381.27778464207773520915277),
	FLIT( 48612189337.342165760569399191505451),
	FLIT( 22750853238.533041145316962210754846),
	FLIT( 10759909800.736022100927428350864552),
	FLIT( 5035014000.2995260339248336242977650),
	FLIT( 2390700936.0723962025355088081258307),
	FLIT( 1117953985.0661735266658664098765381),
	FLIT( 533518885.54834525763160355820191917),
	FLIT( 249118974.61233805307540809865618595),
	FLIT( 119674503.86711416382122413569950774),
	FLIT( 55731833.611605620915410621291603786),
	FLIT( 27007615.310435627946981365891162116),
	FLIT( 12522259.656340321556749351356495515),
	FLIT( 6139385.8152913544239210935229102383),
	FLIT( 2827023.6968710568926027294356815781),
	FLIT( 1407979.5933782963803709279365190951),
	FLIT( 641568.32774174789443613530049279885),
	FLIT( 326433.70222156329746385494881865629),
	FLIT( 146431.99829213393467530701239954174),
	FLIT( 76721.036023984288729554898331110218),
	FLIT( 33630.457055794339178379506939540069),
	FLIT( 18347.170699977025256077766065638457),
	FLIT( 7776.1581023442464729403383849441448),
	FLIT( 4487.2184531889983114533472409858737),
	FLIT( 1811.2054188018712619819390899687162),
	FLIT( 1130.4578581775129886232713059431532),
	FLIT( 425.17984568438211699948244362960518),
	FLIT( 296.43641350761319568192993933485109),
	FLIT( 100.64812354551159119684235730687608),
	FLIT( 82.229156827042532426364264069770890),
	FLIT( 24.038823062292954288537279534011051),
	FLIT( 24.824942121894071094348195261061510),
	FLIT( 5.8004145665998724717337954376216700),
	FLIT( 8.7217825838153462179505692569103341),
	FLIT( 1.4112911430779799505534410562978515),
	FLIT( 4.8583209516339961204939002695085397),
	FLIT( 1.5156034480216573216370580047112014),
	FLIT( 0.13031607421787775137121557811572000e-67),
	};

static const struct RootApprox rootapprox[] = {	/* table of roots */
	{FLIT(-7.00024), FLIT(-7.0001),
	FLIT(-9086776953917897053963482578182625.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(6913260740001846327361025458772383.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr00)},

	{FLIT(-6.99988), FLIT(-6.9998),
	FLIT(-9086261844946976891757708165812213.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(9750753549304090805250118780404071.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr01)},

	{FLIT(-6.0015), FLIT(-6.0011),
	FLIT(-7790243502811557176157561777576162.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(5724445758100794610504675008621123.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(1298074214633706907132624082305024.0),
	ROOTPOLY(sr02)},

	{FLIT(-5.999), FLIT(-5.9982),
	FLIT(-7786637693601862282424884524960892.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(329024724088433828283486538118297.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(81129638414606681695789005144064.0),
	ROOTPOLY(sr03)},

	{FLIT(-5.009), FLIT(-5.007),
	FLIT(-6501038865559612729623708787101268.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(673406285501364980295890810752169.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(162259276829213363391578010288128.0),
	ROOTPOLY(sr04)},

	{FLIT(-4.992), FLIT(-4.991),
	FLIT(-6479395389104072782969547387676194.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(7769753707828124300539074120735565.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(1298074214633706907132624082305024.0),
	ROOTPOLY(sr05)},

	{FLIT(-4.046), FLIT(-4.032),
	FLIT(-5243391447742562865211587478184005.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(8.0),
	FLIT(430883602891049815573321959592399.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(162259276829213363391578010288128.0),
	ROOTPOLY(sr06)},

	{FLIT(-3.9649), FLIT(-3.946),
	FLIT(-10268531044926027830558650614190519.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(3213468811305387775907488774559045.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(1298074214633706907132624082305024.0),
	ROOTPOLY(sr07)},

	{FLIT(-3.175), FLIT(-3.115),
	FLIT(-8161202585564862087365692737842398.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(4165289930304467895318816476756457.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(1298074214633706907132624082305024.0),
	ROOTPOLY(sr08)},

	{FLIT(-2.82), FLIT(-2.78),
	FLIT(-7269215601948758679942694860908135.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(6230756230241793154236595595064115.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr09)},

	{FLIT(-2.78), FLIT(-2.69),
	FLIT(-7133391987426702513289125839376216.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(6504711152360117357897031863587579.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr10)},

	{FLIT(-2.69), FLIT(-2.62),
	FLIT(-6879793337558646607802907636216628.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(4153837486827862102824397063376077.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(1298074214633706907132624082305024.0),
	ROOTPOLY(sr11)},

	{FLIT(-2.62), FLIT(-2.559),
	FLIT(-6724024431802601778946992746340025.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(7061523727607365574801475007739331.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr12)},

	{FLIT(-2.559), FLIT(-2.3514),
	FLIT(-6378800914803110149617797261397252.0)
		/ FLIT(10384593717069655257060992658440192.0) * FLIT(4.0),
	FLIT(4811747745649328600473428485651073.0) 
		/ FLIT(10384593717069655257060992658440192.0) 
		/ FLIT(2596148429267413814265248164610048.0),
	ROOTPOLY(sr13)},
	};

 #else /* FBITS */
  #error lgamma has insufficient precision
 #endif /* FBITS */

#if defined(_BUILD_FOR_RENESAS) && defined(_R_FLT)	/* V10000K102 */
static const FTYPE half_ln2pim1 = FLIT( 4.1893853320467273e-01);
#else
static const FTYPE half_ln2pim1 = FLIT(0.41893853320467274178032973640561765);
#endif												/* V10000K102 */

static FTYPE lgamma_big(FTYPE x)
	{	/* compute ln(gamma(x)), Gamma_big < x */
	FTYPE xinv = FLIT(1.0) / x;
	FTYPE y = FNAME(Log)(x, 0) - FLIT(1.0);
	FTYPE z = x - FLIT(0.5);
	DECLARE(w, xinv)
	short xexp;

	FNAME(Dunscale)(&xexp, &z);
	z *= y;
	if (FNAME(Dscale)(&z, xexp) < 0)
		z += half_ln2pim1 + POLY(xinv, w);
	else
		_Feraise(_FE_OVERFLOW);
	return (z);
	}

#if defined(_BUILD_FOR_RENESAS) && defined(_R_FLT)	/* V10000K102 */
static const FTYPE loge2hi = (FTYPE)(5814539.0 / 8388608.0);
static const FTYPE loge2lo = FLIT( 1.1730463525082348e-07);

static const FTYPE pi = FLIT( 3.1415926535897931e+00);
#else
static const FTYPE loge2hi = (FTYPE)(5814539.0 / 8388608.0);
static const FTYPE loge2lo = FLIT(1.1730463525082348212145817656807550e-7);

static const FTYPE pi = FLIT(3.14159265358979323846264338327950288);
#endif												/* V10000K102 */

FTYPE (FFUN(lgamma))(FTYPE x)
	{	/* compute ln(gamma(x)) */
	FTYPE y = x;

	switch (FNAME(Dint)(&y, 0))
		{	/* test for special codes */
	case _NANCODE:
		return (x);

	case _INFCODE:
		return (FCONST(Inf));

	case 0:
		if (x <= FLIT(0.0))
			{	/* x is zero or a negative integer, huge */
			_Feraise(_FE_DIVBYZERO);
			return (FCONST(Inf));
			}

	default:	/* result finite, may fall through */
		if (x < -FNAME(Gamma_big))
			return (FNAME(Log)(pi / (x * FNAME(Sin)(pi * (x - y), 0)), 0)
				- lgamma_big(-x));
		else if (FNAME(Gamma_big) <= x)
			return (lgamma_big(x));
		else
			{	/* |x| < Gamma_big, finite */
			int i;

			for (i = 0; i < NROOTS; ++i)
				{	/* test if close enough to an approximated root */
				if (rootapprox[i].from < x && x < rootapprox[i].to)
					break;
				}

			if (i < NROOTS)
				{	/* compute root approximation  */
				FTYPE y = x;

				y = ((y - rootapprox[i].zero1) - rootapprox[i].zero2);
				return (FNAME(Poly)(y, rootapprox[i].sr,
					rootapprox[i].srsize));
				}
			else if (FLIT(1.2) < x  && x < FLIT(1.5))
				{	/* compute direct approximation  */
				FTYPE y = x;
				FTYPE ans;
				int i;

				for (i = 0; i < sizeof (approx) / sizeof (approx[0]); ++i)
					if (y < approx[i].to)
						break;

				y -= approx[i].mid;
				ans = FNAME(Poly)(y, approx[i].num, approx[i].nsize)
					/ FNAME(Poly)(y, approx[i].den, approx[i].dsize);

				return (ans);
				}
			else
				{	/* no roots, compute log of tgamma */
				short xexp;
				FTYPE z;

				y = FNAME(Tgamma)(&x, &xexp);
				x = -FNAME(Log)(x < FLIT(0.0) ? -x : x, 0);
				z = -xexp;
				x += loge2lo * z;
				x += loge2hi * z;
				x += FFUN(log1p)(y);
				return (x);
				}
			}
		}
	}
_STD_END

/*
 * Copyright (c) 1992-2007 by P.J. Plauger.  ALL RIGHTS RESERVED.
 * Consult your license regarding permissions and restrictions.
V5.03:0216 */
